#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


from zope.interface import alsoProvides, implementer
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_content.features.renderer.interfaces import IContentRenderer, IFakeVocabularyContent, \
    IRenderedContent, IRendererSettings
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import get_object_factory
from pyams_utils.request import check_request


@implementer(IRenderedContent)
class RenderedContentMixin(object):
    """Renderer mixin interface"""

    renderer = None
    """Attribute used to store selected content renderer.    
    Subclasses should generally override this attribute to define a "Choice" field property based
    on a given renderers vocabulary. 
    """

    renderer_interface = IContentRenderer
    """Content renderer interface"""

    def get_renderer(self, request=None):
        """Get rendering adapter based on selected renderer name"""
        if request is None:
            request = check_request()
        renderer = request.registry.queryMultiAdapter((self, request), self.renderer_interface,
                                                      name=self.renderer or '')
        if (renderer is not None) and ('lang' in request.params):
            renderer.language = request.params['lang']
        return renderer


@adapter_config(context=IRenderedContent, provides=IContentRenderer)
def rendered_content_renderer_factory(context):
    """Rendered content renderer factory"""
    return context.get_renderer()


@adapter_config(context=IRenderedContent, provides=IRendererSettings)
def rendered_content_renderer_settings_factory(context):
    """Rendered content renderer settings factory"""
    renderer = IContentRenderer(context)
    if renderer.settings_interface is None:
        return None
    return renderer.settings_interface(context)


class RenderersVocabulary(SimpleVocabulary):
    """Renderers vocabulary base class"""

    renderer_interface = IContentRenderer
    """Renderer interface"""

    content_interface = IRenderedContent
    """Interface used to check current context"""

    content_factory = None
    """Factory used to create a new context if current context doesn't implements required interface.
    If no factory is given, vocabulary is looking for default object factory for given interface.
    """

    def __init__(self, context=None):
        request = check_request()
        translate = request.localizer.translate
        registry = request.registry
        if not self.content_interface.providedBy(context):
            factory = self.content_factory
            if factory is None:
                factory = get_object_factory(self.content_interface)
            if factory is not None:
                context = factory()
                alsoProvides(context, IFakeVocabularyContent)
        terms = [SimpleTerm(name, title=translate(adapter.label))
                 for name, adapter in sorted(registry.getAdapters((context, request),
                                                                  self.renderer_interface),
                                             key=lambda x: x[1].weight)]
        super(RenderersVocabulary, self).__init__(terms)
