#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from z3c.form import field

from pyams_content.features.redirect import RedirectionRule
from pyams_content.features.redirect.interfaces import IRedirectionManager, IRedirectionManagerTarget, IRedirectionRule
from pyams_content.features.redirect.zmi.container import RedirectionsContainerTable, RedirectionsContainerView
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IFormHelp
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_table_row_refresh_event
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import adapter_config
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.layer import IAdminLayer

from pyams_content import _


@viewlet_config(name='add-rule.action', context=IRedirectionManagerTarget, layer=IPyAMSLayer,
                view=RedirectionsContainerView, manager=IWidgetTitleViewletManager,
                permission=MANAGE_SITE_ROOT_PERMISSION, weight=1)
class RedirectionRuleAddAction(ToolbarAction):
    """Redirection rule add action"""

    label = _("Add rule")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-rule.html'
    modal_target = True


@pagelet_config(name='add-rule.html', context=IRedirectionManagerTarget, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='add-rule.json', context=IRedirectionManagerTarget, layer=IPyAMSLayer, base=AJAXAddForm)
class RedirectionRuleAddForm(AdminDialogAddForm):
    """Redirection rule add form"""

    dialog_class = 'modal-large'
    legend = _("Add new redirection rule")
    icon_css_class = 'fa fa-fw fa-map-signs'

    fields = field.Fields(IRedirectionRule).omit('__parent__', '__name__', 'active', 'chained')
    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    def create(self, data):
        return RedirectionRule()

    def add(self, object):
        IRedirectionManager(self.context).append(object)

    def nextURL(self):
        return absolute_url(self.context, self.request, 'redirections.html')


@pagelet_config(name='properties.html', context=IRedirectionRule, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='properties.json', context=IRedirectionRule, layer=IPyAMSLayer)
class RedirectionRulePropertiesEditForm(AdminDialogEditForm):
    """Redirection rule properties edit form"""

    dialog_class = 'modal-large'
    prefix = 'rule_properties.'

    legend = _("Edit redirection rule properties")
    icon_css_class = 'fa fa-fw fa-map-signs'

    fields = field.Fields(IRedirectionRule).omit('__parent__', '__name__', 'active', 'chained')
    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IRedirectionRule, ())
        if updated:
            target = get_parent(self.context, IRedirectionManagerTarget)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(target, self.request, RedirectionsContainerTable, self.context))
        return output


@adapter_config(context=(IRedirectionManagerTarget, IAdminLayer, RedirectionRuleAddForm), provides=IFormHelp)
@adapter_config(context=(IRedirectionRule, IAdminLayer, RedirectionRulePropertiesEditForm), provides=IFormHelp)
class RedirectionRuleFormHelp(FormHelp):
    """Redirection rule form help"""

    message = _("""URL pattern and target URL are defined by *regular expressions* (see |regexp|).
    
In URL pattern, you can use any valid regular expression element, notably:

- « .* » to match any list of characters 

- « ( ) » to "memorize" parts of the URL which can be replaced into target URL

- special characters (like "+") must be escaped with an « \\\\ ».

In target URL, memorized parts can be reused using « \\\\1 », « \\\\2 » and so on, where given number is
the order of the matching pattern element.

.. |regexp| raw:: html

    <a href="https://docs.python.org/3/library/re.html" target="_blank">Python Regular Expressions</a>
""")
    message_format = 'rest'
