#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.response import Response
from zope.container.ordered import OrderedContainer
from zope.location.interfaces import ISublocations
from zope.location.location import locate
from zope.traversing.interfaces import ITraversable

from pyams_catalog.utils import index_object
from pyams_content.features.redirect.interfaces import IRedirectionManager, IRedirectionManagerTarget, IRedirectionRule, \
    REDIRECT_MANAGER_KEY
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config

from pyams_content import _


@factory_config(IRedirectionManager)
class RedirectManager(OrderedContainer):
    """Redirect manager"""

    last_id = 1

    def append(self, value, notify=True):
        key = str(self.last_id)
        if not notify:
            # pre-locate item to avoid multiple notifications
            locate(value, self, key)
        self[key] = value
        self.last_id += 1
        if not notify:
            # make sure that item is correctly indexed
            index_object(value)

    def get_active_items(self):
        yield from filter(lambda x: IRedirectionRule(x).active, self.values())

    def get_response(self, request):
        target_url = request.path_qs
        for rule in self.get_active_items():
            match = rule.match(target_url)
            if match:
                target_url = rule.rewrite(target_url, request)
                if not rule.chained:
                    response = Response()
                    response.status_code = 301 if rule.permanent else 302
                    response.location = target_url
                    return response

    def test_rules(self, source_url, request, check_inactive_rules=False):
        if check_inactive_rules:
            rules = self.values()
        else:
            rules = self.get_active_items()
        for rule in rules:
            match = rule.match(source_url)
            if match:
                target_url = rule.rewrite(source_url, request)
                yield rule, source_url, target_url
                if not rule.chained:
                    raise StopIteration
                source_url = target_url
            else:
                yield rule, source_url, request.localizer.translate(_("not matching"))


@adapter_config(context=IRedirectionManagerTarget, provides=IRedirectionManager)
def redirection_manager_factory(context):
    """Redirection manager factory"""
    return get_annotation_adapter(context, REDIRECT_MANAGER_KEY, IRedirectionManager,
                                  name='++redirect++')


@adapter_config(name='redirect', context=IRedirectionManagerTarget, provides=ITraversable)
class RedirectionManagerNamespace(ContextAdapter):
    """Redirection manager ++redirect++ namespace"""

    def traverse(self, name, furtherpath=None):
        return IRedirectionManager(self.context)


@adapter_config(name='redirect', context=IRedirectionManagerTarget, provides=ISublocations)
class RedirectManagerSublocations(ContextAdapter):
    """redirection manager sub-locations adapter"""

    def sublocations(self):
        return IRedirectionManager(self.context).values()
