#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from pyams_content.features.header.interfaces import IHeaderTarget, IHeaderSettings, IHeaderRenderer, \
    IHeaderRendererSettings
from pyams_form.interfaces.form import IWidgetForm, IUncheckedEditFormButtons, IInnerSubForm, \
    IWidgetsSuffixViewletsManager, IFormHelp
from pyams_portal.interfaces import MANAGE_TEMPLATE_PERMISSION
from pyams_portal.zmi.interfaces import IPortalContextTemplatePropertiesMenu
from pyams_skin.interfaces import IInnerPage
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.interfaces.inherit import IInheritInfo
from pyams_zmi.layer import IAdminLayer
from z3c.form.interfaces import INPUT_MODE

# import packages
from pyams_content.zmi import pyams_content
from pyams_form.form import AJAXEditForm
from pyams_form.group import NamedWidgetsGroup
from pyams_form.help import FormHelp
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.viewlet.menu import MenuItem, MenuDivider
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config, Viewlet
from pyams_zmi.form import AdminEditForm, InnerAdminEditForm
from pyramid.exceptions import NotFound
from pyramid.response import Response
from pyramid.view import view_config
from z3c.form import field, button
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from zope.interface import implementer, alsoProvides, Interface

from pyams_content import _


@viewlet_config(name='header-settings.divider', context=IHeaderTarget, layer=IPyAMSLayer,
                manager=IPortalContextTemplatePropertiesMenu, permission=MANAGE_TEMPLATE_PERMISSION, weight=99)
class HeaderSettingsDivider(MenuDivider):
    """Header settings menu divider"""


@viewlet_config(name='header-settings.menu', context=IHeaderTarget, layer=IPyAMSLayer,
                manager=IPortalContextTemplatePropertiesMenu, permission=MANAGE_TEMPLATE_PERMISSION, weight=100)
class HeaderSettingsMenu(MenuItem):
    """Header settings menu"""

    label = _("Page header")
    url = '#header-settings.html'


class IHeaderSettingsGroup(Interface):
    """Header settings group marker interface"""


@pagelet_config(name='header-settings.html', context=IHeaderTarget, layer=IPyAMSLayer,
                permission=MANAGE_TEMPLATE_PERMISSION)
@implementer(IWidgetForm, IInnerPage)
class HeaderSettingsEditForm(AdminEditForm):
    """Header settings edit form"""

    @property
    def title(self):
        return self.context.title

    legend = _("Edit header settings")

    def getContent(self):
        return IHeaderSettings(self.context)

    @property
    def fields(self):
        if self.getContent().can_inherit:
            fields = field.Fields(IHeaderSettings).select('no_inherit')
            fields['no_inherit'].widgetFactory = SingleCheckBoxFieldWidget
        else:
            fields = field.Fields(Interface)
        return fields

    edit_permission = MANAGE_TEMPLATE_PERMISSION

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IUncheckedEditFormButtons)
        else:
            return button.Buttons(Interface)

    ajax_handler = 'header-settings.json'

    def updateGroups(self):
        if self.getContent().can_inherit:
            group = NamedWidgetsGroup(self, 'header', self.widgets,
                                      ('no_inherit', ),
                                      legend=_("Don't inherit parent header"),
                                      css_class='inner',
                                      switch=True,
                                      checkbox_switch=True,
                                      checkbox_mode='disable',
                                      checkbox_field=IHeaderSettings['no_inherit'])
            if self.mode == INPUT_MODE:
                group.object_data = {
                    'ams-plugins': 'pyams_content',
                    'ams-plugin-pyams_content-src': get_resource_path(pyams_content),
                    'ams-plugin-pyams_content-async': False,
                    'ams-events-handlers': {
                        'ams.checker.opened': 'PyAMS_content.header.submitEditForm',
                        'ams.checker.closed': 'PyAMS_content.header.submitEditForm'
                    }
                }
                alsoProvides(group, IObjectData)
        else:
            group = NamedWidgetsGroup(self, 'header', self.widgets, (), css_class='inner')
        alsoProvides(group, IHeaderSettingsGroup)
        self.add_group(group)
        super(HeaderSettingsEditForm, self).updateGroups()


@view_config(name='header-settings.json', context=IHeaderTarget, request_type=IPyAMSLayer,
             permission=MANAGE_TEMPLATE_PERMISSION, renderer='json', xhr=True)
class HeaderSettingsAJAXEditForm(AJAXEditForm, HeaderSettingsEditForm):
    """Header settings edit form, JSON renderer"""

    def get_ajax_output(self, changes):
        output = super(HeaderSettingsAJAXEditForm, self).get_ajax_output(changes) or {}
        if 'no_inherit' in changes.get(IInheritInfo, ()):
            if 'autosubmit' in self.request.params:
                output['close_form'] = False
            else:
                output['status'] = 'reload'
        return output


@adapter_config(context=(Interface, IAdminLayer, HeaderSettingsEditForm), provides=IFormHelp)
class HeaderSettingsEditFormHelpAdapter(FormHelp):
    """Header settings edit form help adapter"""

    def __new__(cls, context, request, view):
        if (not view.getContent().can_inherit) or (view.mode != INPUT_MODE):
            return None
        return FormHelp.__new__(cls)

    message = _("""WARNING: Header properties are saved automatically when changing inherit mode!!""")
    message_format = 'rest'


@adapter_config(name='renderer', context=(IHeaderTarget, IPyAMSLayer, IHeaderSettingsGroup), provides=IInnerSubForm)
class HeaderSettingsRendererEditSubform(InnerAdminEditForm):
    """Header settings renderer edit form"""

    legend = None

    fields = field.Fields(IHeaderSettings).select('renderer')
    weight = 1

    edit_permission = MANAGE_TEMPLATE_PERMISSION

    _changes = None

    def __init__(self, context, request, group):
        context = IHeaderSettings(context)
        super(HeaderSettingsRendererEditSubform, self).__init__(context, request, group)

    def getContent(self):
        return IHeaderSettings(self.context)

    def updateWidgets(self, prefix=None):
        super(HeaderSettingsRendererEditSubform, self).updateWidgets(prefix)
        if 'renderer' in self.widgets:
            widget = self.widgets['renderer']
            widget.object_data = {
                'ams-change-handler': 'MyAMS.helpers.select2ChangeHelper',
                'ams-change-stop-propagation': 'true',
                'ams-select2-helper-type': 'html',
                'ams-select2-helper-url': absolute_url(self.context, self.request,
                                                       'get-header-settings-renderer-form.html'),
                'ams-select2-helper-argument': 'form.widgets.renderer',
                'ams-select2-helper-target': '#renderer-settings-helper'
            }
            alsoProvides(widget, IObjectData)

    def get_forms(self, include_self=True):
        if include_self and (self._changes is None) and (self.request.method == 'POST'):
            data, errors = self.extractData()
            if not errors:
                self._changes = self.applyChanges(data)
        for form in super(HeaderSettingsRendererEditSubform, self).get_forms(include_self):
            yield form

    def get_ajax_output(self, changes):
        if not changes:
            changes = self._changes
        if changes:
            return {
                'status': 'success',
                'message': self.request.localizer.translate(self.successMessage)
            }
        else:
            return super(HeaderSettingsRendererEditSubform, self).get_ajax_output(changes)


@adapter_config(name='header-renderer-settings-form',
                context=(IHeaderRendererSettings, IPyAMSLayer, HeaderSettingsRendererEditSubform),
                provides=IInnerSubForm)
@adapter_config(name='header-renderer-settings-form',
                context=(IHeaderTarget, IPyAMSLayer, HeaderSettingsAJAXEditForm),
                provides=IInnerSubForm)
class HeaderSettingsRendererSettingsEditForm(InnerAdminEditForm):
    """Header settings renderer settings edit form"""

    legend = _("Header renderer settings")

    def __new__(cls, context, request, view=None):
        settings = IHeaderRendererSettings(context, None)
        if settings is None:
            return None
        return InnerAdminEditForm.__new__(cls)

    def __init__(self, context, request, view=None):
        context = IHeaderRendererSettings(context)
        super(HeaderSettingsRendererSettingsEditForm, self).__init__(context, request, view)


@viewlet_config(name='header-renderer-settings', context=IHeaderSettings, layer=IPyAMSLayer,
                view=HeaderSettingsRendererEditSubform, manager=IWidgetsSuffixViewletsManager)
@template_config(template='templates/renderer-settings.pt', layer=IPyAMSLayer)
class HeaderSettingsRendererSettingsWidgetsSuffix(Viewlet):
    """Header settings renderer settings viewlet"""

    def render_edit_form(self):
        settings = IHeaderSettings(self.context)
        renderer = settings.get_renderer(self.request)
        if (renderer is None) or (renderer.settings_interface is None):
            return ''
        renderer_settings = IHeaderRendererSettings(settings)
        form = HeaderSettingsRendererSettingsEditForm(renderer_settings, self.request)
        form.update()
        return form.render()


@view_config(name='get-header-settings-renderer-form.html', context=IHeaderSettings,
             request_type=IPyAMSLayer, permission=MANAGE_TEMPLATE_PERMISSION, xhr=True)
def get_header_settings_renderer_form(request):
    """Header settings renderer settings form"""
    renderer_name = request.params.get('form.widgets.renderer')
    if renderer_name is None:
        raise NotFound("No provided renderer argument")
    if not renderer_name:
        renderer_name = ''
    renderer = request.registry.queryMultiAdapter((request.context, request), IHeaderRenderer, name=renderer_name)
    if (renderer is None) or (renderer.settings_interface is None):
        return Response('')
    settings = IHeaderSettings(request.context)
    renderer_settings = IHeaderRendererSettings(settings)
    form = HeaderSettingsRendererSettingsEditForm(renderer_settings, request)
    form.update()
    return Response(form.render())
