#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.encode import urlencode

from pyams_content.component.video.interfaces import IExternalVideoRenderer
from pyams_content.component.video.provider import CustomVideoSettings
from pyams_content.component.video.provider.dailymotion import DailymotionVideoSettings
from pyams_content.component.video.provider.vimeo import VimeoVideoSettings
from pyams_content.component.video.provider.youtube import YoutubeVideoSettings
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_viewlet.viewlet import BaseContentProvider


def time_to_seconds(value):
    """Convert min:sec value to seconds"""
    if value and (':' in value):
        min, sec = value.split(':', 1)
        return str(int(min)*60 + int(sec))
    else:
        return value or ''


def get_playlist_id(settings):
    """Include playlist ID if loop is required"""
    if settings.loop:
        return settings.video_id
    else:
        return None


YOUTUBE_PARAMS = (
    ('start_at', 'start', time_to_seconds),
    ('stop_at', 'end', time_to_seconds),
    ('autoplay', 'autoplay', int),
    ('loop', 'loop', int),
    (None, 'playlist', get_playlist_id),
    ('show_commands', 'controls', int),
    ('hide_branding', 'modestbranding', int),
    ('show_related', 'rel', int),
    ('allow_fullscreen', 'fs', int),
    ('disable_keyboard', 'disablekb', int)
)


DAILYMOTION_PARAMS = (
    ('start_at', 'start', time_to_seconds),
    ('autoplay', 'autoplay', int),
    ('show_info', 'ui-start-screen-info', int),
    ('show_commands', 'controls', int),
    ('ui_theme', 'ui-theme', str),
    ('show_branding', 'ui-logo', int),
    ('show_endscreen', 'endscreen-enable', int),
    ('allow_sharing', 'sharing-enable', int)
)


VIMEO_PARAMS = (
    ('show_title', 'title', int),
    ('show_signature', 'byline', int),
    ('color', 'color', str),
    ('autoplay', 'autoplay', int),
    ('loop', 'loop', int)
)


class BaseExternalVideoRenderer(BaseContentProvider):
    """Base external video renderer"""

    params = ()

    def get_url_params(self):
        settings = self.context
        params = {}
        for attr, param, handler in self.params:
            if attr is None:
                result = handler(settings)
            else:
                result = handler(getattr(settings, attr))
            if result is not None:
                params[param] = result
        return urlencode(params)


@adapter_config(context=(CustomVideoSettings, IPyAMSLayer), provides=IExternalVideoRenderer)
@template_config(template='templates/custom-render.pt', layer=IPyAMSLayer)
class CustomVideoRenderer(BaseExternalVideoRenderer):
    """Custom video renderer"""


@adapter_config(context=(YoutubeVideoSettings, IPyAMSLayer), provides=IExternalVideoRenderer)
@template_config(template='templates/youtube-render.pt', layer=IPyAMSLayer)
class YoutubeVideoRenderer(BaseExternalVideoRenderer):
    """Youtube video renderer"""

    params = YOUTUBE_PARAMS


@adapter_config(context=(DailymotionVideoSettings, IPyAMSLayer), provides=IExternalVideoRenderer)
@template_config(template='templates/dailymotion-render.pt', layer=IPyAMSLayer)
class DailymotionVideoRenderer(BaseExternalVideoRenderer):
    """Dailymotion video renderer"""

    params = DAILYMOTION_PARAMS


@adapter_config(context=(VimeoVideoSettings, IPyAMSLayer), provides=IExternalVideoRenderer)
@template_config(template='templates/vimeo-render.pt', layer=IPyAMSLayer)
class VimeoVideoRenderer(BaseExternalVideoRenderer):
    """Vimeo video renderer"""

    params = VIMEO_PARAMS
