#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library
import re

from persistent import Persistent

# import interfaces
from pyams_content.component.video.interfaces import IExternalVideo, IExternalVideoProvider, IExternalVideoSettings
from pyams_content.component.video.provider.interfaces import IVimeoVideoSettings
from pyams_content.features.checker.interfaces import IContentChecker

# import packages
from pyams_content.component.video import external_video_settings_factory
from pyams_content.features.checker import BaseContentChecker
from pyams_utils.adapter import adapter_config
from pyams_utils.registry import utility_config
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_content import _


VIMEO_BASE_URL = re.compile('https://vimeo.com/([0-9]+)')


@implementer(IVimeoVideoSettings)
class VimeoVideoSettings(Persistent):
    """Vimeo video settings"""

    _video_id = FieldProperty(IVimeoVideoSettings['video_id'])
    show_title = FieldProperty(IVimeoVideoSettings['show_title'])
    show_signature = FieldProperty(IVimeoVideoSettings['show_signature'])
    color = FieldProperty(IVimeoVideoSettings['color'])
    autoplay = FieldProperty(IVimeoVideoSettings['autoplay'])
    loop = FieldProperty(IVimeoVideoSettings['loop'])
    allow_fullscreen = FieldProperty(IVimeoVideoSettings['allow_fullscreen'])
    width = FieldProperty(IVimeoVideoSettings['width'])
    height = FieldProperty(IVimeoVideoSettings['height'])

    @property
    def video_id(self):
        return self._video_id

    @video_id.setter
    def video_id(self, value):
        if value:
            match = VIMEO_BASE_URL.match(value)
            if match:
                value = match.groups()[0]
        self._video_id = value


@utility_config(name='Vimeo', provides=IExternalVideoProvider)
class VimeoVideoProvider(object):
    """Vimeo video provider"""

    settings_interface = IVimeoVideoSettings


@adapter_config(context=IExternalVideo, provides=IVimeoVideoSettings)
def vimeo_video_settings_factory(context):
    """Vimeo video settings factory"""
    if context.provider_name != 'Vimeo':
        return None
    return external_video_settings_factory(context)


@adapter_config(context=VimeoVideoProvider, provides=IExternalVideoSettings)
def vimeo_video_provider_settings_factory(context):
    """Vimeo video provider settings factory"""
    return VimeoVideoSettings()


@adapter_config(context=IVimeoVideoSettings, provides=IContentChecker)
class VimeoVideoSettingsContentChecker(BaseContentChecker):
    """Vimeo video settings content checker"""

    label = _("Vimeo settings")

    def inner_check(self, request):
        return []
