#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.paragraph import BaseParagraph, BaseParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.paragraph.interfaces import IParagraphFactory
from pyams_content.component.video import ExternalVideo, ExternalVideoContentChecker
from pyams_content.component.video.interfaces import EXTERNAL_VIDEO_PARAGRAPH_NAME, EXTERNAL_VIDEO_PARAGRAPH_RENDERERS, \
    EXTERNAL_VIDEO_PARAGRAPH_TYPE, IExternalVideoParagraph
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.json import IJSONConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


@factory_config(provided=IExternalVideoParagraph)
class ExternalVideoParagraph(ExternalVideo, BaseParagraph):
    """External video paragraph"""

    factory_name = EXTERNAL_VIDEO_PARAGRAPH_TYPE
    icon_class = 'fa-youtube-play'
    icon_hint = EXTERNAL_VIDEO_PARAGRAPH_NAME

    renderer = FieldProperty(IExternalVideoParagraph['renderer'])


@utility_config(name=EXTERNAL_VIDEO_PARAGRAPH_TYPE,
                provides=IParagraphFactory)
class ExternalVideoParagraphFactory(BaseParagraphFactory):
    """External video paragraph factory"""

    name = EXTERNAL_VIDEO_PARAGRAPH_NAME
    content_type = ExternalVideoParagraph


@adapter_config(required=IExternalVideoParagraph,
                provides=IContentChecker)
class ExternalVideoParagraphContentChecker(ExternalVideoContentChecker):
    """External video paragraph content checker"""

    @property
    def label(self):
        request = check_request()
        translate = request.localizer.translate
        return II18n(self.context).query_attribute('title', request) or \
            '({0})'.format(translate(self.context.icon_hint).lower())

    def inner_check(self, request):
        output = super(ExternalVideoParagraphContentChecker, self).inner_check(request)
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language, )
        missing_value = translate(MISSING_VALUE)
        missing_lang_value = translate(MISSING_LANG_VALUE)
        i18n = II18n(self.context)
        for attr in ('title', ):
            for lang in langs:
                value = i18n.get_attribute(attr, lang, request)
                if not value:
                    if len(langs) == 1:
                        output.insert(0, missing_value.format(field=translate(IExternalVideoParagraph[attr].title)))
                    else:
                        output.insert(0, missing_lang_value.format(field=translate(IExternalVideoParagraph[attr].title),
                                                                   lang=lang))
        return output


@vocabulary_config(name=EXTERNAL_VIDEO_PARAGRAPH_RENDERERS)
class ExternalVideoParagraphRendererVocabulary(RenderersVocabulary):
    """External video paragraph renderers vocabulary"""

    content_interface = IExternalVideoParagraph


@adapter_config(required=(IExternalVideoParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONExternalVideoParagraphConverter(JSONParagraphConverter):
    """JSON external video paragraph converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        self.get_attribute(self.context, 'provider_name', result, name='provider')
        settings = self.context.settings
        if settings is not None:
            self.get_attribute(settings, 'video_id', result)
        return result
