#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from zope.container.contained import Contained
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_content import _
from pyams_content.component.video.interfaces import IExternalVideo, IExternalVideoProvider, IExternalVideoSettings
from pyams_content.features.checker import BaseContentChecker, VALUE_OK
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_VALUE, MISSING_LANG_VALUE
from pyams_i18n.interfaces import II18nManager, INegotiator, II18n
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.registry import query_utility, get_utility
from pyams_utils.traversing import get_parent


@implementer(IExternalVideo)
class ExternalVideo(Persistent, Contained):
    """External video persistent class"""

    author = FieldProperty(IExternalVideo['author'])
    description = FieldProperty(IExternalVideo['description'])
    provider_name = FieldProperty(IExternalVideo['provider_name'])

    def get_provider(self):
        return query_utility(IExternalVideoProvider, name=self.provider_name or '')

    @property
    def settings(self):
        provider = self.get_provider()
        if provider is None:
            return None
        return provider.settings_interface(self)


EXTERNAL_VIDEO_SETTINGS_KEY = 'pyams_content.video::{0}'


@adapter_config(context=IExternalVideo, provides=IExternalVideoSettings)
def external_video_settings_factory(context):
    """External video settings factory"""
    if not context.provider_name:
        return None
    provider = context.get_provider()
    settings_key = EXTERNAL_VIDEO_SETTINGS_KEY.format(context.provider_name.lower())
    return get_annotation_adapter(context, settings_key,
                                  factory=lambda: IExternalVideoSettings(provider))


@adapter_config(context=IExternalVideo, provides=IContentChecker)
class ExternalVideoContentChecker(BaseContentChecker):
    """External video content checker"""

    label = _("External video")
    weight = 50

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language, )
        missing_value = translate(MISSING_VALUE)
        missing_lang_value = translate(MISSING_LANG_VALUE)
        i18n = II18n(self.context)
        for attr in ('description', ):
            for lang in langs:
                value = i18n.get_attribute(attr, lang, request)
                if not value:
                    if len(langs) == 1:
                        output.append(missing_value.format(field=translate(IExternalVideo[attr].title)))
                    else:
                        output.append(missing_lang_value.format(field=translate(IExternalVideo[attr].title),
                                                                lang=lang))
        for attr in ('author', 'provider_name'):
            value = getattr(self.context, attr)
            if not value:
                output.append(missing_value.format(field=translate(IExternalVideo[attr].title)))
        settings = self.context.settings
        if settings is None:
            pass
        else:
            checker = IContentChecker(settings, None)
            if checker is not None:
                checker_output = checker.inner_check(request)
                if checker_output:
                    output.append('<div class="padding-left-20">')
                    output.append('- {0} :'.format(translate(checker.label)))
                    output.append([checker.sep.join(checker_output)])
                    output.append('</div>')
                else:
                    output.append('- {0} : {1}'.format(translate(checker.label), translate(VALUE_OK)))
        return output
