#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_content.component.verbatim.paragraph module

"""

__docformat__ = 'restructuredtext'

from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.illustration import IBasicIllustrationTarget
from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, \
    BaseParagraphFactory, \
    IParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.verbatim.interfaces import IVerbatimParagraph, \
    VERBATIM_PARAGRAPH_NAME, VERBATIM_PARAGRAPH_RENDERERS, VERBATIM_PARAGRAPH_TYPE
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, \
    MISSING_VALUE
from pyams_content.features.json import IJSONConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


@factory_config(provided=IVerbatimParagraph)
@implementer(IBasicIllustrationTarget)
class VerbatimParagraph(BaseParagraph):
    """Verbatim paragraph"""

    factory_name = VERBATIM_PARAGRAPH_TYPE
    icon_class = 'fa-quote-right'
    icon_hint = VERBATIM_PARAGRAPH_NAME

    quote = FieldProperty(IVerbatimParagraph['quote'])
    author = FieldProperty(IVerbatimParagraph['author'])
    charge = FieldProperty(IVerbatimParagraph['charge'])
    renderer = FieldProperty(IVerbatimParagraph['renderer'])


@utility_config(name=VERBATIM_PARAGRAPH_TYPE,
                provides=IParagraphFactory)
class VerbatimParagraphFactory(BaseParagraphFactory):
    """Verbatim paragraph factory"""

    name = VERBATIM_PARAGRAPH_NAME
    content_type = VerbatimParagraph
    secondary_menu = True


@adapter_config(required=IVerbatimParagraph,
                provides=IContentChecker)
class VerbatimParagraphContentChecker(BaseParagraphContentChecker):
    """Verbatim paragraph content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for lang in langs:
            for attr in ('title', 'quote', 'charge'):
                value = i18n.get_attribute(attr, lang, request)
                if not value:
                    field_title = translate(IVerbatimParagraph[attr].title)
                    if len(langs) == 1:
                        output.append(translate(MISSING_VALUE).format(field=field_title))
                    else:
                        output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        for attr in ('author',):
            value = getattr(self.context, attr, None)
            if not value:
                field_title = translate(IVerbatimParagraph[attr].title)
                output.append(translate(MISSING_VALUE).format(field=field_title))
        return output


@vocabulary_config(name=VERBATIM_PARAGRAPH_RENDERERS)
class VerbatimParagraphRendererVocabulary(RenderersVocabulary):
    """Verbatim paragraph renderers vocabulary"""

    content_interface = IVerbatimParagraph


@adapter_config(required=(IVerbatimParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONVerbatimParagraphConverter(JSONParagraphConverter):
    """JSON verbatim paragraph converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        self.get_i18n_attribute(self.context, 'quote', params.get('lang'), result)
        self.get_attribute(self.context, 'author', result)
        self.get_i18n_attribute(self.context, 'charge', params.get('lang'), result)
        return result
