#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from z3c.form import field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from zope.interface import alsoProvides

from pyams_content import _
from pyams_content.component.theme.interfaces import ICollectionsManager, \
    ICollectionsManagerTarget, ITagsManager, ITagsManagerTarget, IThemesManager, \
    IThemesManagerTarget
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION, MANAGE_TOOL_PERMISSION
from pyams_content.zmi import pyams_content
from pyams_form.form import ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.interfaces.data import IObjectData
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer


#
# Tags management view
#

@viewlet_config(name='tags-manager.menu', context=ITagsManagerTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_SITE_ROOT_PERMISSION, weight=730)
class TagsManagerMenu(MenuItem):
    """Tags menu"""

    label = _("Tags settings...")
    icon_class = 'fa-tag'
    url = 'tags-manager.html'
    modal_target = True


@pagelet_config(name='tags-manager.html', context=ITagsManagerTarget, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='tags.json', context=ITagsManagerTarget, layer=IPyAMSLayer)
class TagsManagerEditForm(AdminDialogEditForm):
    """Tags manager edit form"""

    prefix = 'tags_manager.'

    legend = _("Selected tags")
    dialog_class = 'modal-large'

    fields = field.Fields(ITagsManager)
    fields['enable_glossary'].widgetFactory = SingleCheckBoxFieldWidget

    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    def getContent(self):
        return ITagsManager(self.context)

    def updateWidgets(self, prefix=None):
        # We have to store thesaurus name in request header
        # to be able to set extract name correctly
        # See :ref:`pyams_thesaurus.thesaurus.ThesaurusExtractsVocabulary`
        if self.request.method == 'POST':
            param_name = '{}widgets.thesaurus_name:list'.format(self.prefix)
            param_value = self.request.params.get(param_name)
            if param_value is not None:
                self.request.headers['X-Thesaurus-Name'] = param_value

        super(TagsManagerEditForm, self).updateWidgets(prefix)
        widget = self.widgets['thesaurus_name']
        widget.object_data = {
            'ams-plugins': 'pyams_content',
            'ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'ams-change-handler': 'PyAMS_content.themes.updateThesaurus'
        }
        alsoProvides(widget, IObjectData)

    def updateGroups(self):
        self.add_group(NamedWidgetsGroup(self, 'glossary', self.widgets,
                                         ('enable_glossary', 'glossary_thesaurus_name'),
                                         legend=_("Enable glossary"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_field=ITagsManager['enable_glossary']))
        super(TagsManagerEditForm, self).updateGroups()


#
# Themes management view
#

@viewlet_config(name='themes-manager.menu', context=IThemesManagerTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_TOOL_PERMISSION, weight=740)
class ThemesManagerMenu(MenuItem):
    """Themes menu"""

    label = _("Themes settings...")
    icon_class = 'fa-tags'
    url = 'themes-manager.html'
    modal_target = True


@pagelet_config(name='themes-manager.html', context=IThemesManagerTarget, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='themes-manager.json', context=IThemesManagerTarget, layer=IPyAMSLayer)
class ThemesManagerEditForm(AdminDialogEditForm):
    """Themes manager edit form"""

    prefix = 'themes_manager.'

    legend = _("Selected themes")

    fields = field.Fields(IThemesManager)
    edit_permission = MANAGE_TOOL_PERMISSION

    def getContent(self):
        return IThemesManager(self.context)

    def updateWidgets(self, prefix=None):
        # We have to store thesaurus name in request header
        # to be able to set extract name correctly
        # See :ref:`pyams_thesaurus.thesaurus.ThesaurusExtractsVocabulary`
        if self.request.method == 'POST':
            param_name = '{}widgets.thesaurus_name:list'.format(self.prefix)
            param_value = self.request.params.get(param_name)
            if param_value is not None:
                self.request.headers['X-Thesaurus-Name'] = param_value

        super(ThemesManagerEditForm, self).updateWidgets(prefix)
        widget = self.widgets['thesaurus_name']
        widget.object_data = {
            'ams-plugins': 'pyams_content',
            'ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'ams-change-handler': 'PyAMS_content.themes.updateThesaurus'
        }
        alsoProvides(widget, IObjectData)


#
# Collections management view
#

@viewlet_config(name='collections-manager.menu', context=ICollectionsManagerTarget,
                layer=IAdminLayer, manager=IPropertiesMenu, permission=MANAGE_TOOL_PERMISSION,
                weight=750)
class CollectionsManagerMenu(MenuItem):
    """Collections menu"""

    label = _("Collections settings...")
    icon_class = 'fa-book'
    url = 'collections-manager.html'
    modal_target = True


@pagelet_config(name='collections-manager.html', context=ICollectionsManagerTarget, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='collections.json', context=ICollectionsManagerTarget, layer=IPyAMSLayer)
class CollectionsManagerEditForm(AdminDialogEditForm):
    """Collections manager edit form"""

    prefix = 'collections_manager.'

    legend = _("Selected collections")
    dialog_class = 'modal-large'

    fields = field.Fields(ICollectionsManager)
    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    def getContent(self):
        return ICollectionsManager(self.context)

    def updateWidgets(self, prefix=None):
        # We have to store thesaurus name in request header
        # to be able to set extract name correctly
        # See :ref:`pyams_thesaurus.thesaurus.ThesaurusExtractsVocabulary`
        if self.request.method == 'POST':
            param_name = '{}widgets.thesaurus_name:list'.format(self.prefix)
            param_value = self.request.params.get(param_name)
            if param_value is not None:
                self.request.headers['X-Thesaurus-Name'] = param_value

        super(CollectionsManagerEditForm, self).updateWidgets(prefix)
        widget = self.widgets['thesaurus_name']
        widget.object_data = {
            'ams-plugins': 'pyams_content',
            'ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'ams-change-handler': 'PyAMS_content.themes.updateThesaurus'
        }
        alsoProvides(widget, IObjectData)
