#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from z3c.form import field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from zope.interface import implementer

from pyams_content.component.theme import ICollectionsInfo, ICollectionsManager, \
    ICollectionsTarget, ITagsInfo, ITagsManager, ITagsManagerTarget, ITagsTarget, \
    IThemesInfo, IThemesManager, IThemesManagerTarget, IThemesTarget
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.zmi import WfSharedContentHeaderAdapter
from pyams_form.form import AJAXEditForm, EditForm, ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_form.interfaces.form import IWidgetForm
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces import IDialog, IInnerPage, IPageHeader
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuItem
from pyams_thesaurus.zmi.widget import ThesaurusTermsTreeFieldWidget
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogEditForm, AdminEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Tags management
#

@viewlet_config(name='tags.menu',
                context=ITagsTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=350,
                permission=VIEW_SYSTEM_PERMISSION)
class TagsMenu(MenuItem):
    """Tags menu"""

    label = _("Tags...")
    icon_class = 'fa-tag'
    url = '#tags.html'


class BaseTagsEditForm(EditForm):
    """Base tags edit form"""

    legend = _("Content tags")

    label_css_class = 'control-label hidden'
    input_css_class = 'col-md-12'

    @property
    def fields(self):
        fields = field.Fields(ITagsInfo).select('tags')
        fields['tags'].widgetFactory = ThesaurusTermsTreeFieldWidget
        return fields

    edit_permission = MANAGE_CONTENT_PERMISSION

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        if 'tags' in self.widgets:
            widget = self.widgets['tags']
            target = get_parent(self.context, ITagsManagerTarget)
            if target is not None:
                manager = ITagsManager(target)
                widget.thesaurus_name = manager.thesaurus_name
                widget.extract_name = manager.extract_name


@adapter_config(context=(ITagsTarget, IAdminLayer, BaseTagsEditForm),
                provides=IPageHeader)
class TagsHeaderAdapter(WfSharedContentHeaderAdapter):
    """Shared content tags header adapter"""

    icon_class = 'fa fa-fw fa-tag'


@pagelet_config(name='tags.html',
                context=ITagsTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='tags.json',
             context=ITagsTarget, layer=IPyAMSLayer,
             base=AJAXEditForm)
@implementer(IWidgetForm, IInnerPage)
class TagsEditForm(BaseTagsEditForm, AdminEditForm):
    """Tags edit form"""


@pagelet_config(name='tags-dialog.html',
                context=ITagsTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='tags-dialog.json',
             context=ITagsTarget, layer=IPyAMSLayer,
             base=AJAXEditForm)
@implementer(IDialog)
class TagsDialogEditForm(BaseTagsEditForm, AdminDialogEditForm):
    """Tags dialog edit form"""

    dialog_class = 'modal-large'


#
# Themes management
#

@viewlet_config(name='themes.menu',
                context=IThemesTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=360,
                permission=VIEW_SYSTEM_PERMISSION)
class ThemesMenu(MenuItem):
    """Themes menu"""

    label = _("Themes...")
    icon_class = 'fa-tags'
    url = '#themes.html'


class BaseThemesEditForm(EditForm):
    """Base themes edit form"""

    legend = _("Content themes")

    label_css_class = 'control-label hidden'
    input_css_class = 'col-md-12'

    @property
    def fields(self):
        if IThemesInfo(self.context).can_inherit:
            fields = field.Fields(IThemesInfo).select('no_inherit', 'themes')
            fields['no_inherit'].widgetFactory = SingleCheckBoxFieldWidget
        else:
            fields = field.Fields(IThemesInfo).select('themes')
        fields['themes'].widgetFactory = ThesaurusTermsTreeFieldWidget
        return fields

    edit_permission = MANAGE_CONTENT_PERMISSION

    def updateGroups(self):
        if IThemesInfo(self.context).can_inherit:
            group = NamedWidgetsGroup(self, 'themes', self.widgets,
                                      ('no_inherit', 'themes'),
                                      legend=_("Don't inherit parent themes"),
                                      css_class='inner',
                                      switch=True,
                                      checkbox_switch=True,
                                      checkbox_mode='disable',
                                      checkbox_field=IThemesInfo['no_inherit'])
        else:
            group = NamedWidgetsGroup(self, 'themes', self.widgets, (), css_class='inner')
        self.add_group(group)
        super().updateGroups()

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        if 'themes' in self.widgets:
            widget = self.widgets['themes']
            target = get_parent(self.context, IThemesManagerTarget)
            if target is not None:
                manager = IThemesManager(target)
                widget.thesaurus_name = manager.thesaurus_name
                widget.extract_name = manager.extract_name


@adapter_config(context=(IThemesTarget, IAdminLayer, BaseThemesEditForm),
                provides=IPageHeader)
class ThemesHeaderAdapter(WfSharedContentHeaderAdapter):
    """Shared content themes header adapter"""

    icon_class = 'fa fa-fw fa-tags'


@pagelet_config(name='themes.html',
                context=IThemesTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='themes.json',
             context=IThemesTarget, layer=IPyAMSLayer,
             base=AJAXEditForm)
@implementer(IWidgetForm, IInnerPage)
class ThemesEditForm(BaseThemesEditForm, AdminEditForm):
    """Themes edit form"""


@pagelet_config(name='themes-dialog.html',
                context=IThemesTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='themes-dialog.json',
             context=IThemesTarget, layer=IPyAMSLayer,
             base=AJAXEditForm)
@implementer(IDialog)
class ThemesDialogEditForm(BaseThemesEditForm, AdminDialogEditForm):
    """Themes dialog edit form"""

    dialog_class = 'modal-large'


#
# Collections management
#

@viewlet_config(name='collections.menu',
                context=ICollectionsTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=370,
                permission=VIEW_SYSTEM_PERMISSION)
class CollectionsMenu(MenuItem):
    """Collections menu"""

    label = _("Collections...")
    icon_class = 'fa-book'
    url = '#collections.html'


@pagelet_config(name='collections.html',
                context=ICollectionsTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='collections.json',
             context=ICollectionsTarget, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION)
@implementer(IWidgetForm, IInnerPage)
class CollectionsEditForm(AdminEditForm):
    """Collections edit form"""

    legend = _("Content collections")

    label_css_class = 'control-label hidden'
    input_css_class = 'col-md-12'

    fields = field.Fields(ICollectionsInfo)
    fields['collections'].widgetFactory = ThesaurusTermsTreeFieldWidget

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        if 'collections' in self.widgets:
            widget = self.widgets['collections']
            manager = ICollectionsManager(self.request.root)
            widget.thesaurus_name = manager.thesaurus_name
            widget.extract_name = manager.extract_name


@adapter_config(context=(ICollectionsTarget, IAdminLayer, CollectionsEditForm),
                provides=IPageHeader)
class CollectionsHeaderAdapter(WfSharedContentHeaderAdapter):
    """Shared content collections header adapter"""

    icon_class = 'fa fa-fw fa-book'
