#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from transaction.interfaces import ITransactionManager
from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import implementer

from pyams_content.component.association.zmi.interfaces import IAssociationsParentForm
from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.paragraph.interfaces.video import IVideoParagraph, VIDEO_PARAGRAPH_TYPE
from pyams_content.component.paragraph.video import VideoParagraph
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
    BaseParagraphAddForm, BaseParagraphAddMenu, BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons, \
    get_json_paragraph_refresh_event
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerView, IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_form_refresh_event, get_json_widget_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.interfaces import IPropertiesEditForm

from pyams_content import _


@viewlet_config(name='add-video-paragraph.menu', context=IParagraphContainerTarget, view=IParagraphContainerView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=70)
class VideoParagraphAddMenu(BaseParagraphAddMenu):
    """Video paragraph add menu"""

    label = _("Video paragraph...")
    label_css_class = 'fa fa-fw ' + VideoParagraph.icon_class
    url = 'add-video-paragraph.html'
    paragraph_type = VIDEO_PARAGRAPH_TYPE


@pagelet_config(name='add-video-paragraph.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-video-paragraph.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class VideoParagraphAddForm(BaseParagraphAddForm):
    """Video paragraph add form"""

    legend = _("Add new video paragraph")
    dialog_class = 'modal-large'

    content_interface = IVideoParagraph


@pagelet_config(name='properties.html', context=IVideoParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IVideoParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class VideoParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Video paragraph properties edit form"""

    prefix = 'video_properties.'

    legend = _("Edit video properties")
    dialog_class = 'modal-large'

    content_interface = IVideoParagraph

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IVideoParagraph, ())
        if 'data' in updated:
            # we have to commit transaction to be able to handle blobs...
            ITransactionManager(self.context).get().commit()
            output.setdefault('events', []).append(
                get_json_form_refresh_event(self.context, self.request, VideoParagraphPropertiesInnerEditForm))
        else:
            if 'title' in updated:
                output.setdefault('events', []).append(
                    get_json_paragraph_refresh_event(self.context, self.request))
            if 'renderer' in updated:
                output.setdefault('events', []).append(
                    get_json_widget_refresh_event(self.context, self.request,
                                                  VideoParagraphPropertiesInnerEditForm, 'renderer'))
        return output


@adapter_config(context=(IVideoParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IVideoParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm, IPropertiesEditForm, IAssociationsParentForm)
class VideoParagraphPropertiesInnerEditForm(VideoParagraphPropertiesEditForm):
    """Video paragraph properties inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()
