#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from pyramid.decorator import reify
from pyramid.events import subscriber
from pyramid.view import view_config
from z3c.form import button, field
from z3c.form.interfaces import IDataExtractedEvent, INPUT_MODE
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, Invalid, implementer

from pyams_content.component.paragraph import BaseParagraph
from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.paragraph.interfaces.pictogram import IPictogramContainer, \
    IPictogramContainerTarget, IPictogramItem, IPictogramParagraph, PICTOGRAM_PARAGRAPH_TYPE
from pyams_content.component.paragraph.pictogram import PictogramItem, PictogramParagraph
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, \
    BaseParagraphAJAXEditForm, BaseParagraphAddForm, BaseParagraphAddMenu, \
    BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons
from pyams_content.component.paragraph.zmi import IParagraphContainerView
from pyams_content.component.paragraph.zmi.interfaces import IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.reference.pictograms.zmi.widget import PictogramSelectFieldWidget
from pyams_content.shared.common import IWfSharedContent
from pyams_file.image import render_image
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces.form import IInnerForm, IInnerSubForm
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.column import I18nAttrColumn
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import switch_element_visibility
from pyams_skin.event import get_json_switched_table_refresh_event, \
    get_json_table_row_refresh_event, get_json_widget_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu, IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import BaseTable, I18nColumn, SorterColumn, TrashColumn, \
    VisibilitySwitcherColumn
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.text import get_text_start
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.interfaces import IPropertiesEditForm
from pyams_zmi.zmi.table import InnerTableView


__docformat__ = 'restructuredtext'

from pyams_content import _


class IPictogramsView(Interface):
    """Pictograms view marker interface"""


class IPictogramsParentForm(Interface):
    """Pictograms parent form marker interface"""


@viewlet_config(name='add-pictogram-paragraph.menu', context=IParagraphContainerTarget,
                view=IParagraphContainerView, layer=IPyAMSLayer, manager=IToolbarAddingMenu,
                weight=600)
class PictogramParagraphAddMenu(BaseParagraphAddMenu):
    """Pictogram paragraph add menu"""

    label = _("Pictograms...")
    label_css_class = 'fa fa-fw ' + PictogramParagraph.icon_class
    url = 'add-pictogram-paragraph.html'
    paragraph_type = PICTOGRAM_PARAGRAPH_TYPE


@pagelet_config(name='add-pictogram-paragraph.html', context=IParagraphContainerTarget,
                layer=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-pictogram-paragraph.json', context=IParagraphContainerTarget,
             layer=IPyAMSLayer, base=BaseParagraphAJAXAddForm)
class PictogramParagraphAddForm(BaseParagraphAddForm):
    """Pictogram paragraph add form"""

    legend = _("Add new pictogram paragraph")

    content_interface = IPictogramParagraph


@pagelet_config(name='properties.html', context=IPictogramParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IPictogramParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class PictogramParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Pictogram paragraph properties edit form"""

    prefix = 'pictograms_properties.'

    @property
    def title(self):
        content = get_parent(self.context, IWfSharedContent)
        return II18n(content).query_attribute('title', request=self.request)

    legend = _("Edit pictogram paragraph properties")

    content_interface = IPictogramParagraph

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IPictogramParagraph, ())
        if 'renderer' in updated:
            output.setdefault('events', []).append(
                get_json_widget_refresh_event(self.context, self.request,
                                              PictogramParagraphInnerEditForm, 'renderer'))
        return output


@adapter_config(context=(IPictogramParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IPictogramParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm, IPropertiesEditForm, IPictogramsParentForm)
class PictogramParagraphInnerEditForm(PictogramParagraphPropertiesEditForm):
    """Pictogram paragraph inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()


#
# Pictogram items table view
#

class PictogramsTable(ProtectedFormObjectMixin, BaseTable):
    """Pictograms view inner table"""

    prefix = 'pictograms'

    hide_header = True
    sortOn = None

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes['table'] = {
            'id': self.id,
            'data-ams-location': absolute_url(IPictogramContainer(self.context), self.request),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-pictograms-order.json'
        }
        attributes.setdefault('td', {}).update({
            'data-ams-attribute-switcher': self.get_switcher_target
        })
        return attributes

    @staticmethod
    def get_switcher_target(element, column):
        if column.__name__ == 'show-hide':
            return 'switch-pictogram-visibility.json'

    @reify
    def values(self):
        return list(super().values)


@adapter_config(context=(IPictogramContainerTarget, IPyAMSLayer, PictogramsTable),
                provides=IValues)
class PictogramsTableValuesAdapter(ContextRequestViewAdapter):
    """Pictograms table values adapter"""

    @property
    def values(self):
        return IPictogramContainer(self.context).values()


@adapter_config(name='sorter',
                context=(IPictogramContainerTarget, IPyAMSLayer, PictogramsTable),
                provides=IColumn)
class PictogramsTableSorterColumn(ProtectedFormObjectMixin, SorterColumn):
    """Pictograms table sorter column"""


@view_config(name='set-pictograms-order.json', context=IPictogramContainer,
             request_type=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION,
             renderer='json', xhr=True)
def set_pictograms_order(request):
    """Update pictograms order"""
    order = list(map(str, json.loads(request.params.get('names'))))
    request.context.updateOrder(order)
    return {'status': 'success'}


@adapter_config(name='show-hide',
                context=(IPictogramContainerTarget, IPyAMSLayer, PictogramsTable),
                provides=IColumn)
class PictogramsTableShowHideColumn(ProtectedFormObjectMixin, VisibilitySwitcherColumn):
    """Pictograms container visibility switcher column"""


@view_config(name='switch-pictogram-visibility.json', context=IPictogramContainer,
             request_type=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION,
             renderer='json', xhr=True)
def switch_pictogram_visibility(request):
    """Set pictogram visibility"""
    return switch_element_visibility(request, IPictogramContainer)


@adapter_config(name='image',
                context=(IPictogramContainerTarget, IPyAMSLayer, PictogramsTable),
                provides=IColumn)
class PictogramsTableImageColumn(GetAttrColumn):
    """Pictogram image column"""

    header = ''
    weight = 3

    cssClasses = {'td': 'text-center width-50'}
    dt_sortable = 'false'

    def getValue(self, obj):
        pictogram = obj.pictogram
        if pictogram is not None:
            image = II18n(pictogram).query_attribute('image', request=self.request)
            if image:
                return render_image(image, 48, 48, self.request, timestamp=True)
        return '--'


@adapter_config(name='name',
                context=(IPictogramContainerTarget, IPyAMSLayer, PictogramsTable),
                provides=IColumn)
class PictogramsTableNameColumn(I18nColumn, I18nAttrColumn):
    """Pictograms table name column"""

    _header = _("pictogram-item-header", default="Header")
    attrName = 'label'
    weight = 10

    def getValue(self, obj):
        value = super(PictogramsTableNameColumn, self).getValue(obj)
        if not value:
            pictogram = obj.pictogram
            if pictogram is not None:
                value = II18n(pictogram).query_attribute('header', request=self.request)
        return value or BaseParagraph.empty_title


@adapter_config(name='body',
                context=(IPictogramContainerTarget, IPyAMSLayer, PictogramsTable),
                provides=IColumn)
class PictogramsTableBodyColumn(I18nColumn, I18nAttrColumn):
    """Pictograms table body column"""

    _header = _("Associated text")
    attrName = 'body'
    weight = 20

    def getValue(self, obj):
        value = super(PictogramsTableBodyColumn, self).getValue(obj)
        if not value:
            return BaseParagraph.empty_title
        return get_text_start(value, 80, 10)


@adapter_config(name='trash',
                context=(IPictogramContainerTarget, IPyAMSLayer, PictogramsTable),
                provides=IColumn)
class PictogramsTableTrashColumn(ProtectedFormObjectMixin, TrashColumn):
    """Pictograms table trash column"""


@adapter_config(name='pictograms',
                context=(IPictogramContainerTarget, IPyAMSLayer, IPictogramsParentForm),
                provides=IInnerSubForm)
@implementer(IPictogramsView)
class PictogramsView(InnerTableView):
    """Pictograms view"""

    title = _("Pictograms")

    table_class = PictogramsTable
    weight = 100


#
# Pictograms forms
#

@viewlet_config(name='add-pictogram.action', context=IPictogramContainerTarget,
                layer=IPyAMSLayer, view=IPictogramsView, manager=IWidgetTitleViewletManager,
                permission=MANAGE_CONTENT_PERMISSION, weight=1)
class PictogramAddAction(ToolbarAction):
    """Pictogram add action"""

    label = _("Add pictogram")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-pictogram.html'
    modal_target = True


@pagelet_config(name='add-pictogram.html', context=IPictogramContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-pictogram.json', context=IPictogramContainerTarget, layer=IPyAMSLayer,
             base=AJAXAddForm)
class PictogramAddForm(AdminDialogAddForm):
    """Pictogram add form"""

    legend = _("Add new pictogram")
    icon_css_class = 'fa fa-fw fa-arrow-h'

    fields = field.Fields(IPictogramItem).omit('__parent__', '__name__', 'visible')
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def updateWidgets(self, prefix=None):
        super(PictogramAddForm, self).updateWidgets(prefix)
        if 'body' in self.widgets:
            self.widgets['body'].widget_css_class = 'input height-100'

    def create(self, data):
        return PictogramItem()

    def add(self, object):
        IPictogramContainer(self.context).append(object)

    def get_ajax_output(self, changes):
        return {
            'status': 'success',
            'message': self.request.localizer.translate(_("Pictogram was correctly added")),
            'events': [get_json_switched_table_refresh_event(self.context, self.request,
                                                             PictogramsTable), ]
        }


@subscriber(IDataExtractedEvent, form_selector=PictogramAddForm)
def handle_pictogram_add_form_data_extraction(event):
    """Handle pictogram add form data extraction"""
    data = event.data
    if not data.get('pictogram_name'):
        event.form.widgets.errors += (Invalid(_("You must select a pictogram!")), )


@pagelet_config(name='properties.html', context=IPictogramItem, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IPictogramItem, layer=IPyAMSLayer)
class PictogramPropertiesEditForm(AdminDialogEditForm):
    """Pictogram properties edit form"""

    prefix = 'pictogram_properties.'

    legend = _("Edit pictogram properties")
    icon_css_class = 'fa fa-fw fa-linode'

    fields = field.Fields(IPictogramItem).omit('__parent__', '__name__', 'visible')
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def updateWidgets(self, prefix=None):
        super(PictogramPropertiesEditForm, self).updateWidgets(prefix)
        if 'body' in self.widgets:
            self.widgets['body'].widget_css_class = 'input height-100'

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IPictogramItem, ())
        if updated:
            target = get_parent(self.context, IPictogramContainerTarget)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(target, self.request,
                                                 PictogramsTable, self.context))
        return output


@subscriber(IDataExtractedEvent, form_selector=PictogramPropertiesEditForm)
def handle_pictogram_edit_form_data_extraction(event):
    """Handle pictogram edit form data extraction"""
    data = event.data
    if not data.get('pictogram_name'):
        event.form.widgets.errors += (Invalid(_("You must select a pictogram!")), )
