#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import implementer

from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.paragraph.interfaces.keypoint import IKeypointsParagraph, KEYPOINTS_PARAGRAPH_TYPE
from pyams_content.component.paragraph.keypoint import KeypointsParagraph
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
    BaseParagraphAddForm, BaseParagraphAddMenu, BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerView, IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_widget_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_viewlet.viewlet import viewlet_config

from pyams_content import _


@viewlet_config(name='add-keypoints-paragraph.menu', context=IParagraphContainerTarget, view=IParagraphContainerView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=600)
class KeypointsParagraphAddMenu(BaseParagraphAddMenu):
    """Key points paragraph add menu"""

    label = _("Key points...")
    label_css_class = 'fa fa-fw ' + KeypointsParagraph.icon_class
    url = 'add-keypoints-paragraph.html'
    paragraph_type = KEYPOINTS_PARAGRAPH_TYPE


@pagelet_config(name='add-keypoints-paragraph.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-keypoints-paragraph.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class KeypointsParagraphAddForm(BaseParagraphAddForm):
    """Key points paragraph add form"""

    legend = _("Add new key points paragraph")

    content_interface = IKeypointsParagraph

    def updateWidgets(self, prefix=None):
        super(KeypointsParagraphAddForm, self).updateWidgets(prefix)
        if 'body' in self.widgets:
            self.widgets['body'].widget_css_class = 'input height-100'


@pagelet_config(name='properties.html', context=IKeypointsParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IKeypointsParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class KeypointsParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Key points paragraph properties edit form"""

    prefix = 'keypoints_properties.'

    legend = _("Edit key points paragraph properties")

    content_interface = IKeypointsParagraph

    def updateWidgets(self, prefix=None):
        super(KeypointsParagraphPropertiesEditForm, self).updateWidgets(prefix)
        if 'body' in self.widgets:
            self.widgets['body'].widget_css_class = 'input height-100'

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IKeypointsParagraph, ())
        if 'renderer' in updated:
            form = KeypointsParagraphInnerEditForm(self.context, self.request)
            form.update()
            output.setdefault('events', []).append(
                get_json_widget_refresh_event(self.context, self.request, KeypointsParagraphInnerEditForm, 'renderer'))
        return output


@adapter_config(context=(IKeypointsParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IKeypointsParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm)
class KeypointsParagraphInnerEditForm(KeypointsParagraphPropertiesEditForm):
    """Key points paragraph inner edit form"""

    legend = None
    label_css_class = 'control-label col-md-2'
    input_css_class = 'col-md-10'

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()
