#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import Interface, implementer

from pyams_content.component.association.interfaces import IAssociationContainerTarget
from pyams_content.component.association.zmi import AssociationsTable
from pyams_content.component.association.zmi.interfaces import IAssociationsParentForm
from pyams_content.component.paragraph.frame import FrameParagraph
from pyams_content.component.paragraph.interfaces import IBaseParagraph, \
    IParagraphContainerTarget, IParagraphTitle
from pyams_content.component.paragraph.interfaces.frame import FRAME_PARAGRAPH_TYPE, \
    IFrameParagraph
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, \
    BaseParagraphAJAXEditForm, BaseParagraphAddForm, BaseParagraphAddMenu, \
    BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons, \
    get_json_paragraph_refresh_event, get_json_paragraph_toolbar_refresh_event
from pyams_content.component.paragraph.zmi.container import ParagraphContainerTable, \
    ParagraphTitleToolbarViewletManager
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerView, \
    IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.zmi import pyams_content
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_switched_table_refresh_event
from pyams_skin.interfaces.tinymce import ITinyMCEConfiguration
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.html import html_to_text
from pyams_utils.text import get_text_start
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.interfaces import IPropertiesEditForm


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Frame paragraph form interface and custom HTML edit widget
#

class IFrameParagraphForm(Interface):
    """Marker interface for framed text paragraph form"""


@adapter_config(context=(IFrameParagraphForm, IPyAMSLayer), provides=ITinyMCEConfiguration)
class FrameParagraphBodyEditorConfiguration(ContextRequestAdapter):
    """Custom configuration for 'body' widget editor"""

    configuration = {
        'ams-plugins': 'pyams_content',
        'ams-plugin-pyams_content-src': get_resource_path(pyams_content),
        'ams-plugin-pyams_content-async': 'false',
        'ams-tinymce-init-callback': 'PyAMS_content.TinyMCE.initEditor',
        'ams-tinymce-menubar': False,
        'ams-tinymce-plugins': ['paste', 'textcolor', 'lists', 'charmap',
                                'internal_links', 'link'],
        'ams-tinymce-toolbar': 'undo redo | pastetext | bold italic superscript | '
                               'forecolor backcolor | bullist numlist | '
                               'charmap internal_links link',
        'ams-tinymce-toolbar1': False,
        'ams-tinymce-toolbar2': False,
        'ams-tinymce-height': 150
    }


#
# Framed text paragraph
#

@adapter_config(context=(IFrameParagraph, IPyAMSLayer), provides=IParagraphTitle)
def frame_paragraph_title_adapter(context, request):
    """Frame paragraph title adapter"""
    i18n = II18n(context)
    title = i18n.query_attribute('title', request=request)
    if not title:
        body = i18n.query_attribute('body', request=request)
        if body:
            title = '<i>{0}</i>'.format(get_text_start(html_to_text(body), 50, 10))
    return title


@viewlet_config(name='add-frame-paragraph.menu', context=IParagraphContainerTarget, view=IParagraphContainerView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=600)
class FrameParagraphAddMenu(BaseParagraphAddMenu):
    """Framed text paragraph add menu"""

    label = _("Framed text...")
    label_css_class = 'fa fa-fw ' + FrameParagraph.icon_class
    url = 'add-frame-paragraph.html'
    paragraph_type = FRAME_PARAGRAPH_TYPE


@pagelet_config(name='add-frame-paragraph.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-frame-paragraph.json', context=IParagraphContainerTarget, request_type=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
@implementer(IFrameParagraphForm)
class FrameParagraphAddForm(BaseParagraphAddForm):
    """Framed text paragraph add form"""

    legend = _("Add new framed text paragraph")
    dialog_class = 'modal-large'

    label_css_class = 'control-label col-md-2'
    input_css_class = 'col-md-10'

    content_interface = IFrameParagraph


@pagelet_config(name='properties.html', context=IFrameParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IFrameParagraph, request_type=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IFrameParagraphForm)
class FrameParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Framed text paragraph properties edit form"""

    prefix = 'frame_properties.'

    legend = _("Edit framed text paragraph properties")
    dialog_class = 'modal-large'

    label_css_class = 'control-label col-md-2'
    input_css_class = 'col-md-10'

    content_interface = IFrameParagraph

    def updateWidgets(self, prefix=None):
        super(FrameParagraphPropertiesEditForm, self).updateWidgets(prefix)
        if 'body' in self.widgets:
            body_widget = self.widgets['body']
            for lang in body_widget.langs:
                widget = body_widget.widgets[lang]
                widget.id = '{id}_{name}'.format(id=widget.id, name=self.context.__name__)

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if 'body' in changes.get(IFrameParagraph, ()):
            if 'title' not in changes.get(IBaseParagraph, ()):
                output.setdefault('events', []).append(
                    get_json_paragraph_refresh_event(self.context, self.request))
            # refresh associations count markers
            parent = get_parent(self.context, IAssociationContainerTarget)
            output.setdefault('events', []).append(
                get_json_paragraph_toolbar_refresh_event(parent, self.request,
                                                         ParagraphContainerTable,
                                                         ParagraphTitleToolbarViewletManager))
            # refresh associations table
            output.setdefault('events', []).append(
                get_json_switched_table_refresh_event(self.context, self.request,
                                                      AssociationsTable))
        return output


@adapter_config(context=(IFrameParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IFrameParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm, IPropertiesEditForm, IAssociationsParentForm)
class FrameParagraphInnerEditForm(FrameParagraphPropertiesEditForm):
    """Framed text paragraph inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()
