#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from pyramid.decorator import reify
from pyramid.exceptions import NotFound
from pyramid.view import view_config
from z3c.form import field
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.contentprovider.interfaces import IContentProvider
from zope.interface import Interface, implementer

from pyams_content.component.association.interfaces import IAssociationContainer
from pyams_content.component.association.zmi import AssociationsContainerView
from pyams_content.component.paragraph import BaseParagraph, IParagraphTitle
from pyams_content.component.paragraph.interfaces import IBaseParagraph, IParagraphContainer, \
    IParagraphContainerTarget, IParagraphFactorySettings
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerTable, \
    IParagraphContainerView, IParagraphInnerEditor, IParagraphTitleToolbar
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION, PUBLISH_CONTENT_PERMISSION
from pyams_content.shared.common.zmi import WfModifiedContentColumnMixin
from pyams_content.zmi import pyams_content
from pyams_form.interfaces.form import IFormSecurityContext, IInnerSubForm
from pyams_form.security import ProtectedFormObjectMixin, get_checker
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import switch_element_attribute, switch_element_visibility
from pyams_skin.interfaces import IInnerPage, IPageHeader
from pyams_skin.interfaces.container import ITableElementEditor
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.page import DefaultPageHeaderAdapter
from pyams_skin.table import AttributeSwitcherColumn, BaseTable, I18nColumn, ImageColumn, \
    SorterColumn, TrashColumn, VisibilitySwitcherColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_template.template import template_config
from pyams_utils.adapter import ContextRequestViewAdapter, NullAdapter, adapter_config
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.manager import TemplateBasedViewletManager, WeightOrderedViewletManager, \
    viewletmanager_config
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from pyams_zmi.form import AdminDialogDisplayForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView, ContainerAdminView
from pyams_zmi.zmi.table import InnerTableView

__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='paragraphs.menu',
                context=IParagraphContainerTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION, weight=100)
class ParagraphsContainerMenu(MenuItem):
    """Paragraphs container menu"""

    label = _("Contents...")
    icon_class = 'fa-paragraph'
    url = '#paragraphs.html'


#
# Paragraphs container view
#

@implementer(IParagraphContainerTable)
class ParagraphContainerBaseTable(ProtectedFormObjectMixin, BaseTable):
    """Paragraphs container table"""

    prefix = 'paragraphs'

    hide_header = True
    sortOn = None

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-plugins': 'pyams_content',
            'data-ams-plugin-pyams_content-src': get_resource_path(pyams_content),
            'data-ams-location': absolute_url(IParagraphContainer(self.context), self.request),
            'data-ams-pre-reload': 'PyAMS_content.paragraphs.preReload',
            'data-ams-post-reload': 'PyAMS_content.paragraphs.postReload',
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-paragraphs-order.json',
            'data-ams-anchor-icon-on': 'fa fa-fw fa-anchor',
            'data-ams-anchor-icon-off': 'fa fa-fw fa-anchor txt-color-silver opacity-50',
            'data-ams-locked-icon-on': 'fa fa-fw fa-lock',
            'data-ams-locked-icon-off': 'fa fa-fw fa-lock txt-color-silver opacity-50'
        })
        attributes.setdefault('td', {}).update({
            'data-ams-attribute-switcher': self.get_switcher_target,
            'data-ams-switcher-attribute-name': self.get_switcher_attribute
        })
        return attributes

    @staticmethod
    def get_switcher_target(element, column):
        if column.__name__ == 'show-hide':
            return 'switch-paragraph-visibility.json'
        elif column.__name__ == 'anchor':
            return 'switch-paragraph-anchor.json'
        elif column.__name__ == 'lock':
            return 'switch-paragraph-lock.json'

    @staticmethod
    def get_switcher_attribute(element, column):
        if column.__name__ == 'show-hide':
            return 'visible'
        elif column.__name__ == 'anchor':
            return 'anchor'
        elif column.__name__ == 'lock':
            return 'locked'


class ParagraphContainerTable(ParagraphContainerBaseTable):
    """Paragraph container base table"""

    @reify
    def values(self):
        return list(super(ParagraphContainerBaseTable, self).values)

    def render(self):
        if not self.values:
            translate = self.request.localizer.translate
            message = translate(_("No currently defined paragraph."))
            permission = self.permission
            if (not permission) or self.request.has_permission(permission, context=self.context):
                manager = get_parent(self.context, IParagraphFactorySettings)
                if (manager is not None) and not manager.allowed_paragraphs:
                    message = '{0}<br />{1}'.format(message,
                                                    translate(_("Check allowed paragraph types to be able to create "
                                                                "new paragraphs.")))
            return self.renderEmptyTable(message)
        return super(ParagraphContainerBaseTable, self).render()


@adapter_config(context=(IParagraphContainerTarget, IPyAMSLayer, ParagraphContainerBaseTable),
                provides=IValues)
class ParagraphContainerValues(ContextRequestViewAdapter):
    """Paragraphs container values"""

    @property
    def values(self):
        return IParagraphContainer(self.context).values()


@adapter_config(context=(IBaseParagraph, IPyAMSLayer, ParagraphContainerTable),
                provides=ITableElementEditor)
class BaseParagraphTableElementEditor(NullAdapter):
    """Base paragraph table element editor"""


@adapter_config(name='sorter',
                context=(IParagraphContainerTarget, IPyAMSLayer, ParagraphContainerBaseTable),
                provides=IColumn)
class ParagraphContainerSorterColumn(ProtectedFormObjectMixin, SorterColumn):
    """Paragraphs container sorter column"""


@view_config(name='set-paragraphs-order.json',
             context=IParagraphContainer, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def set_paragraphs_order(request):
    """Update paragraphs order"""
    order = list(map(str, json.loads(request.params.get('names'))))
    request.context.updateOrder(order)
    return {'status': 'success'}


@adapter_config(name='show-hide',
                context=(IParagraphContainerTarget, IPyAMSLayer, ParagraphContainerBaseTable),
                provides=IColumn)
class ParagraphContainerShowHideColumn(ProtectedFormObjectMixin, VisibilitySwitcherColumn):
    """Paragraphs container visibility switcher column"""


@view_config(name='switch-paragraph-visibility.json',
             context=IParagraphContainer, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def switch_paragraph_visibility(request):
    """Switch paragraph visibility"""
    return switch_element_visibility(request, IParagraphContainer)


@adapter_config(context=ParagraphContainerShowHideColumn, provides=IFormSecurityContext)
def showhide_column_security_context_factory(column):
    """Show/hide column security context factory"""
    return column.table.context


@adapter_config(name='anchor',
                context=(IParagraphContainerTarget, IPyAMSLayer, ParagraphContainerBaseTable),
                provides=IColumn)
class ParagraphContainerAnchorColumn(ProtectedFormObjectMixin, AttributeSwitcherColumn):
    """Paragraphs container anchor switcher column"""

    switch_attribute = 'anchor'

    on_icon_class = 'fa fa-fw fa-anchor'
    off_icon_class = 'fa fa-fw fa-anchor txt-color-silver opacity-50'

    icon_hint = _("Set navigation anchor")

    weight = 6


@view_config(name='switch-paragraph-anchor.json',
             context=IParagraphContainer, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def switch_paragraph_anchor(request):
    """Switch paragraph anchor"""
    return switch_element_attribute(request, IParagraphContainer, attribute_name='anchor')


@adapter_config(context=ParagraphContainerAnchorColumn, provides=IFormSecurityContext)
def anchor_column_security_context_factory(column):
    """Anchor column security context factory"""
    return column.table.context


@adapter_config(name='pictogram',
                context=(IParagraphContainerTarget, IPyAMSLayer, ParagraphContainerBaseTable),
                provides=IColumn)
class ParagraphContainerPictogramColumn(ImageColumn):
    """Paragraph container pictogram column"""

    weight = 8

    def get_icon_class(self, item):
        return item.icon_class

    def get_icon_hint(self, item):
        return self.request.localizer.translate(item.icon_hint)


@viewletmanager_config(name='pyams_paragraph.title_toolbar',
                       context=IBaseParagraph, layer=IPyAMSLayer,
                       view=ParagraphContainerTable, provides=IParagraphTitleToolbar)
@template_config(template='templates/paragraph-title-toolbar.pt', layer=IPyAMSLayer)
@implementer(IParagraphTitleToolbar)
class ParagraphTitleToolbarViewletManager(TemplateBasedViewletManager, WeightOrderedViewletManager):
    """Paragraph title toolbar viewlet manager"""


def get_paragraph_title_hints(item, request, table):
    """Get paragraphs column title hints"""
    registry = request.registry
    provider = registry.queryMultiAdapter((item, request, table), IContentProvider,
                                          name='pyams_paragraph.title_toolbar')
    if provider is not None:
        provider.update()
        return provider.render()


@adapter_config(name='name',
                context=(IParagraphContainerTarget, IPyAMSLayer, ParagraphContainerBaseTable),
                provides=IColumn)
class ParagraphContainerBaseTitleColumn(I18nColumn, WfModifiedContentColumnMixin, GetAttrColumn):
    """Paragraph container base title column"""

    _header = _("Title")

    weight = 50

    def renderCell(self, item):
        return '<span class="title">{0}</span>'.format(super(ParagraphContainerBaseTitleColumn, self).renderCell(item))

    def getValue(self, obj):
        adapter = self.request.registry.queryMultiAdapter((obj, self.request), IParagraphTitle)
        if adapter is not None:
            return adapter
        return II18n(obj).query_attribute('title', request=self.request) or BaseParagraph.empty_title


@adapter_config(name='name',
                context=(IParagraphContainerTarget, IPyAMSLayer, ParagraphContainerTable),
                provides=IColumn)
class ParagraphContainerTitleColumn(ParagraphContainerBaseTitleColumn):
    """Paragraph container title column"""

    _header = _("Show/hide all paragraphs")

    def renderHeadCell(self):
        return '<span data-ams-stop-propagation="true"' \
               '      data-ams-click-handler="PyAMS_content.paragraphs.switchAllEditors">' \
               '    <span class="small hint" title="{hint}" data-ams-hint-gravity="e">' \
               '        <i class="fa fa-plus-square-o switch"></i>' \
               '    </span>&nbsp;&nbsp;&nbsp;{title}' \
               '</span>'.format(
            hint=self.request.localizer.translate(_("Click to open/close all paragraphs editors")),
            title=super(ParagraphContainerTitleColumn, self).renderHeadCell())

    def renderCell(self, item):
        provider = get_paragraph_title_hints(item, self.request, self.table) or ''
        return '<div data-ams-stop-propagation="true" ' \
               '     data-ams-click-handler="PyAMS_content.paragraphs.switchEditor">{provider}' \
               '    <span class="small hint" title="{hint}" data-ams-hint-gravity="e">' \
               '        <i class="fa fa-plus-square-o switch"></i>' \
               '    </span>&nbsp;&nbsp;&nbsp;<span class="title">{title}</span>' \
               '</div>' \
               '<div class="inner-table-form editor margin-x-10 margin-bottom-0"></div>'.format(
            provider=provider,
            hint=self.request.localizer.translate(_("Click to open/close paragraph editor")),
            title=super(ParagraphContainerTitleColumn, self).renderCell(item))


@template_config(template='templates/paragraph-title-icon.pt', layer=IPyAMSLayer)
class ParagraphContainerCounterBase(Viewlet):
    """Paragraph container base counter viewlet"""

    weight = 0

    marker_type = None
    count = None


@adapter_config(name='lock',
                context=(IParagraphContainerTarget, IPyAMSLayer, ParagraphContainerBaseTable),
                provides=IColumn)
class ParagraphContainerLockedColumn(AttributeSwitcherColumn):
    """Paragraphs container lock switcher column"""

    permission = PUBLISH_CONTENT_PERMISSION
    switch_attribute = 'locked'

    on_icon_class = 'fa fa-fw fa-lock'
    off_icon_class = 'fa fa-fw fa-lock txt-color-silver opacity-50'

    weight = 9000

    def get_icon_hint(self, item):
        if not self.has_permission(item):
            if item.locked:
                hint = _("This paragraph is locked and can't be removed!")
            else:
                hint = _("This paragraph is not locked")
        else:
            hint = _("Click to lock/unlock paragraph")
        return self.request.localizer.translate(hint)

    def renderCell(self, item):
        if not self.has_permission(item):
            return '''<span class="hint" title="{title}" data-ams-hint-gravity="e">
                {icon}
            </span>'''.format(title=self.get_icon_hint(item),
                              icon=self.get_icon(item))
        return super().renderCell(item)


@view_config(name='switch-paragraph-lock.json',
             context=IParagraphContainer, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def switch_paragraph_locked(request):
    """Switch paragraph lock"""
    return switch_element_attribute(request, IParagraphContainer, attribute_name='locked')


@adapter_config(name='trash',
                context=(IParagraphContainerTarget, IPyAMSLayer, ParagraphContainerBaseTable),
                provides=IColumn)
class ParagraphContainerTrashColumn(ProtectedFormObjectMixin, TrashColumn):
    """Paragraphs container trash column"""

    action_type = 'delete'

    def has_permission(self, item):
        checker = get_checker(self.request, item, action=self.action_type)
        permission = checker.edit_permission if checker is not None else None
        if permission is not None:
            return self.request.has_permission(permission, context=item)
        return super().has_permission(item)


@pagelet_config(name='paragraphs.html',
                context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IParagraphContainerView)
class ParagraphContainerView(ContainerAdminView):
    """Paragraphs container view"""

    title = _("Content blocks")
    table_class = ParagraphContainerTable


@pagelet_config(name='paragraphs-dialog.html',
                context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class ParagraphContainerDialogView(AdminDialogDisplayForm):
    """Paragraphs container dialog view"""

    title = _("Content blocks")
    dialog_class = 'modal-large no-widget-toolbar'
    fieldset_class = 'height-300'

    fields = field.Fields(Interface)


@adapter_config(name='paragraphs',
                context=(IParagraphContainerTarget, IPyAMSLayer, ParagraphContainerDialogView),
                provides=IInnerSubForm)
@implementer(IParagraphContainerView)
class ParagraphsView(InnerTableView):
    """Paragraphs view"""

    title = _("Content blocks")

    table_class = ParagraphContainerBaseTable
    weight = 90


@adapter_config(context=(IParagraphContainerTarget, IPyAMSLayer, ParagraphContainerView),
                provides=IPageHeader)
class ParagraphHeaderAdapter(DefaultPageHeaderAdapter):
    """Paragraphs container header adapter"""

    back_url = '#properties.html'
    icon_class = 'fa fa-fw fa-paragraph'


@view_config(name='get-paragraph-editor.json',
             context=IParagraphContainer, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION, renderer='json', xhr=True)
def get_paragraph_editor(request):
    """Get inner editor for a given paragraph"""
    container = IParagraphContainer(request.context)
    paragraph = container.get(str(request.params.get('object_name')))
    if paragraph is None:
        raise NotFound()
    registry = request.registry
    editor = registry.queryMultiAdapter((paragraph, request), IParagraphInnerEditor)
    if editor is None:
        editor = registry.queryAdapter(paragraph, IParagraphInnerEditor)
    if editor is not None:
        editor.update()
        return editor.render()


@view_config(name='get-paragraphs-editors.json',
             context=IParagraphContainer, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION, renderer='json', xhr=True)
def get_paragraphs_editors(request):
    """Get all paragraphs inner editors"""
    container = IParagraphContainer(request.context)
    registry = request.registry
    result = {}
    for key, paragraph in container.items():
        editor = registry.queryMultiAdapter((paragraph, request), IParagraphInnerEditor)
        if editor is None:
            editor = registry.queryAdapter(paragraph, IParagraphInnerEditor)
        if editor is not None:
            editor.update()
            result[key] = editor.render()
    return result


#
# Paragraphs associations view
#

@viewlet_config(name='paragraphs-associations.menu',
                context=IParagraphContainerTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION, weight=110)
class ParagraphsAssociationsMenu(MenuItem):
    """Paragraphs associations container menu"""

    label = _("Links and attachments...")
    icon_class = 'fa-link'
    url = '#paragraphs-associations.html'


@pagelet_config(name='paragraphs-associations.html',
                context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@template_config(template='templates/associations.pt', layer=IPyAMSLayer)
@implementer(IInnerPage)
class ParagraphsAssociationsView(AdminView):
    """Paragraphs associations view"""

    title = _("Content blocks links and attachments")

    @reify
    def associations(self):
        result = []
        for paragraph in IParagraphContainer(self.context).values():
            associations = IAssociationContainer(paragraph, None)
            if associations is not None:
                view = AssociationsContainerView(paragraph, self.request)
                view.widget_icon_class = 'fa fa-fw {0}'.format(paragraph.icon_class)
                view.title = II18n(paragraph).query_attribute('title', request=self.request) or \
                             BaseParagraph.empty_title
                result.append(view)
        return result

    def update(self):
        super(ParagraphsAssociationsView, self).update()
        for association in self.associations:
            association.update()
