#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from transaction.interfaces import ITransactionManager
from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import implementer

from pyams_content.component.paragraph.contact import ContactParagraph
from pyams_content.component.paragraph.interfaces import IBaseParagraph, IParagraphContainerTarget
from pyams_content.component.paragraph.interfaces.contact import CONTACT_PARAGRAPH_TYPE, IContactParagraph
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
    BaseParagraphAddForm, BaseParagraphAddMenu, BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons, \
    get_json_paragraph_refresh_event
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerView, IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_form_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_viewlet.viewlet import viewlet_config

from pyams_content import _


@viewlet_config(name='add-contact-paragraph.menu', context=IParagraphContainerTarget, view=IParagraphContainerView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=600)
class ContactParagraphAddMenu(BaseParagraphAddMenu):
    """Contact paragraph add menu"""

    label = _("Contact card...")
    label_css_class = 'fa fa-fw ' + ContactParagraph.icon_class
    url = 'add-contact-paragraph.html'
    paragraph_type = CONTACT_PARAGRAPH_TYPE


@pagelet_config(name='add-contact-paragraph.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-contact-paragraph.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class ContactParagraphAddForm(BaseParagraphAddForm):
    """Contact paragraph add form"""

    legend = _("Add new contact card")
    dialog_class = 'modal-large'

    content_interface = IContactParagraph


@pagelet_config(name='properties.html', context=IContactParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IContactParagraph, request_type=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class ContactParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Contact paragraph properties edit form"""

    prefix = 'contact_properties.'

    legend = _("Edit contact card properties")

    content_interface = IContactParagraph

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IBaseParagraph, ())
        if 'title' in updated:
            output.setdefault('events', []).append(get_json_paragraph_refresh_event(self.context, self.request))
        updated = changes.get(IContactParagraph, ())
        if ('photo' in updated) or ('renderer' in updated):
            # we have to commit transaction to be able to handle blobs...
            if 'photo' in updated:
                ITransactionManager(self.context).get().commit()
            output.setdefault('events', []).append(get_json_form_refresh_event(self.context, self.request,
                                                                               ContactParagraphInnerEditForm))
        return output


@adapter_config(context=(IContactParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IContactParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm)
class ContactParagraphInnerEditForm(ContactParagraphPropertiesEditForm):
    """Contact paragraph inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()
