#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.interface import Interface, alsoProvides

from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.paragraph.portlet import IParagraphNavigationPortletSettings
from pyams_content.component.paragraph.portlet.interfaces import \
    IParagraphContainerPortletSettings
from pyams_form.form import AJAXEditForm
from pyams_form.interfaces.form import IInnerTabForm
from pyams_pagelet.interfaces import IPagelet
from pyams_pagelet.pagelet import pagelet_config
from pyams_portal.interfaces import IPortletPreviewer
from pyams_portal.portlet import PortletPreviewer
from pyams_portal.zmi.portlet import PortletSettingsEditor, PortletSettingsPropertiesEditor
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.traversing import get_parent


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Paragraphs container portlet
#

@pagelet_config(name='properties.html',
                context=IParagraphContainerPortletSettings, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class ParagraphContainerPortletSettingsEditor(PortletSettingsEditor):
    """Paragraph container portlet settings editor"""

    settings = IParagraphContainerPortletSettings


@adapter_config(name='properties',
                context=(Interface, IPyAMSLayer, ParagraphContainerPortletSettingsEditor),
                provides=IInnerTabForm)
class ParagraphContainerPortletSettingsPropertiesEditor(PortletSettingsPropertiesEditor):
    """Paragraph container portlet settings editor"""

    @property
    def fields(self):
        fields = super(ParagraphContainerPortletSettingsPropertiesEditor, self).fields
        container = get_parent(self.context, IParagraphContainerTarget)
        if container is None:
            fields = fields.omit('paragraphs')
        return fields

    def updateWidgets(self, prefix=None):
        super(ParagraphContainerPortletSettingsPropertiesEditor, self).updateWidgets(prefix)
        if 'paragraphs' in self.widgets:
            widget = self.widgets['paragraphs']
            widget.object_data = {
                'ams-select2-placeholder': self.request.localizer.translate(
                    _("No filter, all paragraphs selected"))
            }
            alsoProvides(widget, IObjectData)
        if 'factories' in self.widgets:
            widget = self.widgets['factories']
            widget.object_data = {
                'ams-select2-placeholder': self.request.localizer.translate(
                    _("No filter, all paragraph types selected"))
            }
            alsoProvides(widget, IObjectData)


@adapter_config(name='properties.json',
                context=(IParagraphContainerPortletSettings, IPyAMSLayer),
                provides=IPagelet)
class ParagraphContainerPortletConfigurationAJAXEditor(AJAXEditForm,
                                                       ParagraphContainerPortletSettingsEditor):
    """Paragraph container portlet settings editor, JSON renderer"""


@adapter_config(context=(Interface, IPyAMSLayer, Interface, IParagraphContainerPortletSettings),
                provides=IPortletPreviewer)
@template_config(template='templates/container-preview.pt', layer=IPyAMSLayer)
class ParagraphContainerPortletPreviewer(PortletPreviewer):
    """Paragraph container portlet previewer"""


#
# Paragraphs navigation portlet
#

@pagelet_config(name='properties.html',
                context=IParagraphNavigationPortletSettings, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class ParagraphNavigationPortletSettingsEditor(PortletSettingsEditor):
    """Paragraphs navigation portlet settings editor"""

    settings = IParagraphNavigationPortletSettings


@adapter_config(name='properties',
                context=(Interface, IPyAMSLayer, ParagraphNavigationPortletSettingsEditor),
                provides=IInnerTabForm)
class ParagraphNavigationPortletSettingsPropertiesEditor(PortletSettingsPropertiesEditor):
    """Paragraphs navigation portlet settings editor"""

    @property
    def fields(self):
        fields = super(ParagraphNavigationPortletSettingsPropertiesEditor, self).fields
        container = get_parent(self.context, IParagraphContainerTarget)
        if container is None:
            fields = fields.omit('paragraphs')
        return fields

    def updateWidgets(self, prefix=None):
        super(ParagraphNavigationPortletSettingsPropertiesEditor, self).updateWidgets(prefix)
        if 'paragraphs' in self.widgets:
            widget = self.widgets['paragraphs']
            widget.object_data = {
                'ams-select2-placeholder': self.request.localizer.translate(
                    _("No filter, all paragraphs selected"))
            }
            alsoProvides(widget, IObjectData)
        if 'factories' in self.widgets:
            widget = self.widgets['factories']
            widget.object_data = {
                'ams-select2-placeholder': self.request.localizer.translate(
                    _("No filter, all paragraph types selected"))
            }
            alsoProvides(widget, IObjectData)


@adapter_config(name='properties.json',
                context=(IParagraphNavigationPortletSettings, IPyAMSLayer),
                provides=IPagelet)
class ParagraphNavigationPortletConfigurationAJAXEditor(AJAXEditForm,
                                                        ParagraphNavigationPortletSettingsEditor):
    """Paragraphs navigation portlet settings editor, JSON renderer"""


@adapter_config(context=(Interface, IPyAMSLayer, Interface, IParagraphNavigationPortletSettings),
                provides=IPortletPreviewer)
@template_config(template='templates/navigation-preview.pt', layer=IPyAMSLayer)
class ParagraphNavigationPortletPreviewer(PortletPreviewer):
    """Paragraphs navigation portlet previewer"""
