#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.links import InternalReferenceMixin
from pyams_content.component.paragraph import BaseParagraph
from pyams_content.component.paragraph.interfaces import IParagraphContainer, \
    IParagraphContainerTarget, IParagraphFactory
from pyams_content.component.paragraph.portlet.interfaces import \
    IParagraphContainerPortletSettings, IParagraphNavigationPortletSettings
from pyams_i18n.interfaces import II18n
from pyams_portal.portlet import Portlet, PortletSettings, portlet_config
from pyams_utils.factory import factory_config
from pyams_utils.interfaces import VIEW_PERMISSION
from pyams_utils.registry import get_global_registry
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent

from pyams_content import _


#
# Paragraphs container portlet
#

PARAGRAPH_CONTAINER_PORTLET_NAME = 'pyams_content.portlet.paragraphs'


@factory_config(provided=IParagraphContainerPortletSettings)
class ParagraphContainerPortletSettings(InternalReferenceMixin, PortletSettings):
    """Shared content portlet persistent settings"""

    title = FieldProperty(IParagraphContainerPortletSettings['title'])
    button_label = FieldProperty(IParagraphContainerPortletSettings['button_label'])
    paragraphs = FieldProperty(IParagraphContainerPortletSettings['paragraphs'])
    factories = FieldProperty(IParagraphContainerPortletSettings['factories'])
    anchors_only = FieldProperty(IParagraphContainerPortletSettings['anchors_only'])
    exclude_anchors = FieldProperty(IParagraphContainerPortletSettings['exclude_anchors'])
    display_navigation_links = FieldProperty(
        IParagraphContainerPortletSettings['display_navigation_links'])
    limit = FieldProperty(IParagraphContainerPortletSettings['limit'])

    def get_paragraphs_labels(self):
        if not self.paragraphs:
            yield '--'
        else:
            target = get_parent(self, IParagraphContainerTarget)
            if target is not None:
                container = IParagraphContainer(target)
                for name in self.paragraphs:
                    paragraph = container.get(name)
                    if name is not None:
                        yield II18n(paragraph).query_attribute('title') or BaseParagraph.empty_title

    def get_paragraph_types_labels(self):
        if not self.factories:
            yield '--'
        else:
            request = check_request()
            registry = get_global_registry()
            for factory_name in self.factories:
                factory = registry.queryUtility(IParagraphFactory, name=factory_name)
                if factory is not None:
                    yield request.localizer.translate(factory.name)


@portlet_config(permission=VIEW_PERMISSION)
class ParagraphContainerPortlet(Portlet):
    """Shared content portlet"""

    name = PARAGRAPH_CONTAINER_PORTLET_NAME
    label = _("Content paragraphs")

    toolbar_css_class = 'fa fa-fw fa-2x fa-paragraph'

    settings_factory = IParagraphContainerPortletSettings


#
# Paragraphs navigation portlet
#

PARAGRAPH_NAVIGATION_PORTLET_NAME = 'pyams_content.portlet.paragraphs.navigation'


@factory_config(provided=IParagraphNavigationPortletSettings)
class ParagraphNavigationPortletSettings(PortletSettings):
    """Paragraph navigation portlet settings"""

    paragraphs = FieldProperty(IParagraphNavigationPortletSettings['paragraphs'])
    factories = FieldProperty(IParagraphNavigationPortletSettings['factories'])
    anchors_only = FieldProperty(IParagraphNavigationPortletSettings['anchors_only'])

    def get_paragraphs_labels(self):
        if not self.paragraphs:
            yield '--'
        else:
            target = get_parent(self, IParagraphContainerTarget)
            if target is not None:
                container = IParagraphContainer(target)
                for name in self.paragraphs:
                    paragraph = container.get(name)
                    if name is not None:
                        yield II18n(paragraph).query_attribute('title') or BaseParagraph.empty_title

    def get_paragraph_types_labels(self):
        if not self.factories:
            yield '--'
        else:
            request = check_request()
            registry = get_global_registry()
            for factory_name in self.factories:
                factory = registry.queryUtility(IParagraphFactory, name=factory_name)
                if factory is not None:
                    yield request.localizer.translate(factory.name)


@portlet_config(permission=VIEW_PERMISSION)
class ParagraphNavigationPortlet(Portlet):
    """Paragraphs navigation portlet"""

    name = PARAGRAPH_NAVIGATION_PORTLET_NAME
    label = _("Paragraphs navigation anchors")

    toolbar_css_class = 'fa fa-fw fa-2x fa-arrows-v'

    settings_factory = IParagraphNavigationPortletSettings
