#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from pyramid.events import subscriber
from zope.container.contained import Contained
from zope.container.ordered import OrderedContainer
from zope.interface import implementer
from zope.lifecycleevent import ObjectModifiedEvent
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectModifiedEvent, \
    IObjectRemovedEvent
from zope.location import locate
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty
from zope.traversing.interfaces import ITraversable

from pyams_catalog.utils import index_object
from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, \
    BaseParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.paragraph.interfaces import IParagraphContainer, \
    IParagraphContainerTarget, IParagraphFactory
from pyams_content.component.paragraph.interfaces.milestone import IMilestone, \
    IMilestoneContainer, IMilestoneContainerTarget, IMilestoneParagraph, MILESTONE_CONTAINER_KEY, \
    MILESTONE_PARAGRAPH_NAME, MILESTONE_PARAGRAPH_RENDERERS, MILESTONE_PARAGRAPH_TYPE
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import ERROR_VALUE, IContentChecker, \
    MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.json import JSONBaseConverter
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_content.reference.pictograms import IPictogramTable
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import ContextAdapter, ContextRequestAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_current_registry, get_global_registry, get_utility, utility_config, query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config
from pyams_utils.zodb import volatile_property


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Milestone class and adapters
#

@implementer(IMilestone)
class Milestone(Persistent, Contained):
    """Milestone persistent class"""

    visible = FieldProperty(IMilestone['visible'])
    title = FieldProperty(IMilestone['title'])
    label = FieldProperty(IMilestone['label'])
    _pictogram_name = FieldProperty(IMilestone['pictogram_name'])
    anchor = FieldProperty(IMilestone['anchor'])

    @property
    def pictogram_name(self):
        return self._pictogram_name

    @pictogram_name.setter
    def pictogram_name(self, value):
        if value != self._pictogram_name:
            self._pictogram_name = value
            del self.pictogram

    @volatile_property
    def pictogram(self):
        table = query_utility(IPictogramTable)
        if table is not None:
            return table.get(self._pictogram_name)

    @property
    def target(self):
        container = get_parent(self, IParagraphContainer)
        if container is not None:
            return container.get(self.anchor)


@adapter_config(context=IMilestone, provides=IFormContextPermissionChecker)
class MilestonePermissionChecker(ContextAdapter):
    """Milestone permission checker"""

    @property
    def edit_permission(self):
        content = get_parent(self.context, IMilestoneContainerTarget)
        return IFormContextPermissionChecker(content).edit_permission


@subscriber(IObjectAddedEvent, context_selector=IMilestone)
def handle_added_milestone(event):
    """Handle added milestone"""
    content = get_parent(event.object, IMilestoneContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectModifiedEvent, context_selector=IMilestone)
def handle_modified_milestone(event):
    """Handle modified milestone"""
    content = get_parent(event.object, IMilestoneContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectRemovedEvent, context_selector=IMilestone)
def handle_removed_milestone(event):
    """Handle removed milestone"""
    content = get_parent(event.object, IMilestoneContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@adapter_config(context=IMilestone, provides=IContentChecker)
class MilestoneContentChecker(BaseContentChecker):
    """Milestone content checker"""

    @property
    def label(self):
        request = check_request()
        return II18n(self.context).query_attribute('title', request=request)

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for lang in langs:
            for attr in ('title', 'label'):
                value = i18n.get_attribute(attr, lang, request)
                if not value:
                    field_title = translate(IMilestone[attr].title)
                    if len(langs) == 1:
                        output.append(translate(MISSING_VALUE).format(field=field_title))
                    else:
                        output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        field_title = translate(IMilestone['anchor'].title)
        if not self.context.anchor:
            output.append(translate(MISSING_VALUE).format(field=field_title))
        else:
            target = get_parent(self.context, IParagraphContainerTarget)
            if target is not None:
                container = IParagraphContainer(target)
                paragraph = container.get(self.context.anchor)
                if paragraph is None:
                    output.append(translate(ERROR_VALUE).format(field=field_title,
                                                                message=translate(_("Selected paragraph is missing"))))
                elif not paragraph.visible:
                    output.append(translate(ERROR_VALUE).format(field=field_title,
                                                                message=translate(_("Selected paragraph is not "
                                                                                    "visible"))))
        return output


@adapter_config(required=(IMilestone, IPyAMSLayer),
                provides=IJSONConverter)
class JSONMilestoneConverter(JSONBaseConverter):
    """JSON milestone converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        self.get_i18n_attribute(self.context, 'title', params.get('lang'), result)
        self.get_i18n_attribute(self.context, 'label', params.get('lang'), result)
        return result


#
# Milestones container classes and adapters
#

@factory_config(IMilestoneContainer)
class MilestoneContainer(OrderedContainer):
    """Milestones container"""

    last_id = 1

    def append(self, value, notify=True):
        key = str(self.last_id)
        if not notify:
            # pre-locate milestone item to avoid multiple notifications
            locate(value, self, key)
        self[key] = value
        self.last_id += 1
        if not notify:
            # make sure that milestone item is correctly indexed
            index_object(value)

    def get_visible_items(self):
        yield from filter(lambda x: IMilestone(x).visible, self.values())


@adapter_config(context=IMilestoneContainerTarget, provides=IMilestoneContainer)
def milestone_container_factory(target):
    """Milestone container factory"""
    return get_annotation_adapter(target, MILESTONE_CONTAINER_KEY, IMilestoneContainer,
                                  name='++milestones++')


@adapter_config(name='milestones', context=IMilestoneContainerTarget, provides=ITraversable)
class MilestoneContainerNamespace(ContextAdapter):
    """Milestones container ++milestones++ namespace"""

    def traverse(self, name, furtherpaath=None):
        return IMilestoneContainer(self.context)


@adapter_config(name='milestones', context=IMilestoneContainerTarget, provides=ISublocations)
class MilestoneContainerSublocations(ContextAdapter):
    """Milestones container sub-locations adapter"""

    def sublocations(self):
        return IMilestoneContainer(self.context).values()


@adapter_config(name='milestones', context=IMilestoneContainerTarget, provides=IContentChecker)
class MilestoneContainerContentChecker(BaseContentChecker):
    """Milestones container content checker"""

    label = MILESTONE_PARAGRAPH_NAME
    sep = '\n'
    weight = 200

    def inner_check(self, request):
        output = []
        registry = request.registry
        for milestone in IMilestoneContainer(self.context).values():
            if not milestone.visible:
                continue
            for name, checker in sorted(registry.getAdapters((milestone,), IContentChecker),
                                        key=lambda x: x[1].weight):
                output.append('- {0} :'.format(II18n(milestone).query_attribute('title', request=request)))
                output.append(checker.get_check_output(request))
        return output


@factory_config(provided=IMilestoneParagraph)
class MilestoneParagraph(BaseParagraph):
    """Milestones paragraph"""

    factory_name = MILESTONE_PARAGRAPH_TYPE
    icon_class = 'fa-arrows-h'
    icon_hint = MILESTONE_PARAGRAPH_NAME

    body = FieldProperty(IMilestoneParagraph['body'])
    renderer = FieldProperty(IMilestoneParagraph['renderer'])


@utility_config(name=MILESTONE_PARAGRAPH_TYPE, provides=IParagraphFactory)
class MilestoneParagraphFactory(BaseParagraphFactory):
    """Milestones paragraph factory"""

    name = MILESTONE_PARAGRAPH_NAME
    content_type = MilestoneParagraph


@adapter_config(context=IMilestoneParagraph, provides=IContentChecker)
class MilestoneParagraphContentChecker(BaseParagraphContentChecker):
    """Milestones paragraph content checker"""

    @property
    def label(self):
        request = check_request()
        translate = request.localizer.translate
        return II18n(self.context).query_attribute('title', request) or \
               '({0})'.format(translate(self.context.icon_hint).lower())

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('title', lang, request)
            if not value:
                field_title = translate(IMilestoneParagraph['title'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        return output


@vocabulary_config(name=MILESTONE_PARAGRAPH_RENDERERS)
class MilestoneParagraphRendererVocabulary(RenderersVocabulary):
    """Milestones paragraph renderers vocabulary"""

    content_interface = IMilestoneParagraph


@adapter_config(required=(IMilestoneParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONMilestoneParagraphConverter(JSONParagraphConverter):
    """JSON milestone paragraph converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        self.get_i18n_attribute(self.context, 'body', params.get('lang'), result)
        self.get_list_attribute(IMilestoneContainer(self.context).get_visible_items(),
                                params, result, name='items')
        return result
