#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.annotation.interfaces import IAttributeAnnotatable
from zope.container.constraints import containers, contains
from zope.container.interfaces import IOrderedContainer
from zope.contentprovider.interfaces import IContentProvider
from zope.interface import Attribute, Interface
from zope.schema import Bool, Choice, List, Set

from pyams_content.features.renderer.interfaces import IRenderedContent
from pyams_i18n.schema import I18nTextLineField


__docformat__ = 'restructuredtext'

from pyams_content import _


PARAGRAPH_CONTAINER_KEY = 'pyams_content.paragraph'


class IBaseParagraph(IRenderedContent, IAttributeAnnotatable):
    """Base paragraph interface"""

    containers('.IParagraphContainer')

    factory_name = Attribute("Name of paragraph factory")
    icon_class = Attribute("Icon class in paragraphs list")
    icon_hint = Attribute("Icon hint in paragraphs list")

    visible = Bool(title=_("Visible?"),
                   description=_("Is this paragraph visible in front-office?"),
                   required=True,
                   default=True)

    anchor = Bool(title=_("Anchor?"),
                  description=_("Is this paragraph a navigation anchor?"),
                  required=True,
                  default=False)

    locked = Bool(title=_("Locked?"),
                  description=_("A locked paragraph can only be removed by a content manager "
                                "or a webmaster"),
                  required=True,
                  default=False)

    title = I18nTextLineField(title=_("§ Title"),
                              required=False)


PARAGRAPH_HIDDEN_FIELDS = ('__parent__', '__name__', 'visible', 'anchor', 'locked')


class IParagraphTitle(Interface):
    """Paragraph title adapter"""


class IParagraphContainer(IOrderedContainer):
    """Paragraphs container"""

    contains(IBaseParagraph)

    def append(self, value):
        """Add given value to container"""

    def get_visible_paragraphs(self, names=None, anchors_only=False, exclude_anchors=False,
                               factories=None, limit=None):
        """Get visible paragraphs matching given arguments"""


CONTENT_PARAGRAPHS_VOCABULARY = 'PyAMS content paragraphs'


class IParagraphContainerTarget(IAttributeAnnotatable):
    """Paragraphs container marker interface"""


PARAGRAPH_FACTORIES_VOCABULARY = 'PyAMS paragraph factories'


class IParagraphFactory(Interface):
    """Paragraph factory utility interface"""

    name = Attribute("Factory name")
    content_type = Attribute("Factory content type")
    custom_menu = Attribute("Display factory in 'custom' paragraphs menu")


class IParagraphFactorySettings(Interface):
    """Paragraph factory settings interface

    This interface is used to defined default auto-created paragraphs
    for a given shared tool."""

    allowed_paragraphs = Set(title=_("Allowed paragraphs"),
                             description=_("List of paragraphs allowed for this content type"),
                             required=False,
                             value_type=Choice(vocabulary=PARAGRAPH_FACTORIES_VOCABULARY))

    auto_created_paragraphs = List(title=_("Default paragraphs types"),
                                   description=_("Empty paragraphs of these types will be added "
                                                 "automatically to new contents of this content "
                                                 "type"),
                                   required=False,
                                   value_type=Choice(vocabulary=PARAGRAPH_FACTORIES_VOCABULARY))


class IParagraphRenderer(IContentProvider):
    """Paragraph renderer interface"""

    language = Attribute("Preview language")
