#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.paragraph.interfaces.header import HEADER_PARAGRAPH_NAME, HEADER_PARAGRAPH_RENDERERS, \
    HEADER_PARAGRAPH_TYPE, IHeaderParagraph
from pyams_content.features.checker.interfaces import ERROR_VALUE, IContentChecker
from pyams_content.features.renderer import RenderersVocabulary
from pyams_i18n.interfaces import II18n
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.text import get_text_start
from pyams_utils.vocabulary import vocabulary_config

from pyams_content import _


@factory_config(provided=IHeaderParagraph)
class HeaderParagraph(BaseParagraph):
    """Header paragraph"""

    factory_name = HEADER_PARAGRAPH_TYPE
    icon_class = 'fa-download fa-rotate-180'
    icon_hint = HEADER_PARAGRAPH_NAME

    @property
    def title(self):
        header = II18n(self).query_attribute('header')
        return get_text_start(header, 50, 10)

    header = FieldProperty(IHeaderParagraph['header'])
    renderer = FieldProperty(IHeaderParagraph['renderer'])


@adapter_config(context=IHeaderParagraph, provides=IContentChecker)
class HeaderParagraphContentChecker(BaseParagraphContentChecker):
    """Header paragraph content checker"""

    def inner_check(self, request):
        translate = request.localizer.translate
        field_title = translate(IHeaderParagraph['header'].title)
        return [translate(ERROR_VALUE).format(field=field_title,
                                              message=_("This paragraph type is deprecated and should be removed!")), ]


@vocabulary_config(name=HEADER_PARAGRAPH_RENDERERS)
class HeaderParagraphRendererVocabulary(RenderersVocabulary):
    """Header paragraph renderers vocabulary"""

    content_interface = IHeaderParagraph


@adapter_config(required=(IHeaderParagraph, IPyAMSLayer),
                provides=IPyAMSLayer)
class JSONHeaderParagraphConverter(JSONParagraphConverter):
    """JSON header paragraph converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        self.get_i18n_attribute(self.context, 'header', params.get('lang'), result)
        return result
