#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyams_utils.factory import factory_config
from ZODB.interfaces import IBroken
from zope.location.interfaces import ISublocations
from zope.traversing.interfaces import ITraversable

from pyams_content.component.paragraph.interfaces import IParagraphFactory, IParagraphContainer, \
    IParagraphContainerTarget, PARAGRAPH_CONTAINER_KEY
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import IContentChecker
from pyams_utils.adapter import adapter_config, ContextAdapter, get_annotation_adapter
from pyams_utils.container import BTreeOrderedContainer
from pyams_utils.registry import get_global_registry

from pyams_content import _


@factory_config(IParagraphContainer)
class ParagraphContainer(BTreeOrderedContainer):
    """Paragraphs container"""

    last_id = 1

    def append(self, value):
        key = str(self.last_id)
        self[key] = value
        self.last_id += 1

    def get_visible_paragraphs(self, names=None, anchors_only=False, exclude_anchors=False,
                               factories=None, limit=None):
        count = 0
        if names:
            for name in names:
                paragraph = self.get(name)
                if (paragraph is not None) and \
                        (not IBroken.providedBy(paragraph)) and \
                        paragraph.visible:
                    yield paragraph
                    count += 1
                    if limit and (count == limit):
                        raise StopIteration
        else:
            for paragraph in self.values():
                if IBroken.providedBy(paragraph) or not paragraph.visible:
                    continue
                if anchors_only and not paragraph.anchor:
                    continue
                if exclude_anchors and paragraph.anchor:
                    continue
                if factories:
                    registry = get_global_registry()
                    has_factory = False
                    for factory_name in factories:
                        factory = registry.queryUtility(IParagraphFactory, name=factory_name)
                        has_factory = (factory is not None) and \
                                      isinstance(paragraph, factory.content_type)
                        if has_factory:
                            break
                    if not has_factory:
                        continue
                yield paragraph
                count += 1
                if limit and (count == limit):
                    raise StopIteration


@adapter_config(context=IParagraphContainerTarget, provides=IParagraphContainer)
def paragraph_container_factory(target):
    """Paragraphs container factory"""
    return get_annotation_adapter(target, PARAGRAPH_CONTAINER_KEY, IParagraphContainer,
                                  name='++paras++')


@adapter_config(name='paras', context=IParagraphContainerTarget, provides=ITraversable)
class ParagraphContainerNamespace(ContextAdapter):
    """++paras++ namespace adapter"""

    def traverse(self, name, furtherpath=None):
        return IParagraphContainer(self.context)


@adapter_config(name='paras', context=IParagraphContainerTarget, provides=ISublocations)
class ParagraphContainerSublocations(ContextAdapter):
    """Paragraphs container sublocations"""

    def sublocations(self):
        return IParagraphContainer(self.context).values()


@adapter_config(name='paras', context=IParagraphContainerTarget, provides=IContentChecker)
class ParagraphContainerChecker(BaseContentChecker):
    """Paragraphs container checker"""

    label = _("Paragraphs")
    sep = '\n'
    weight = 10

    def inner_check(self, request):
        output = []
        registry = request.registry
        translate = request.localizer.translate
        visible = False
        for paragraph in IParagraphContainer(self.context).values():
            if not paragraph.visible:
                continue
            visible = True
            for name, checker in sorted(registry.getAdapters((paragraph,), IContentChecker),
                                        key=lambda x: x[1].weight):
                if name:
                    output.append('<div class="padding-left-20">')
                output.append('- {0} :'.format(translate(checker.label)))
                output.append(checker.get_check_output(request))
                if name:
                    output.append('</div>')
        if not visible:
            output.append('<span class="text-danger">{0}</span>'.format(
                translate(_("no visible paragraph"))))
        return output
