#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from html import escape

from pyramid.encode import url_quote, urlencode
from pyramid.events import subscriber
from zope.interface import alsoProvides, directlyProvidedBy, implementer, noLongerProvides
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectModifiedEvent
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_content.component.association import AssociationItem
from pyams_content.component.association.interfaces import IAssociationContainer, \
    IAssociationContainerTarget, IAssociationInfo
from pyams_content.component.links.interfaces import IBaseLink, ICustomInternalLinkTarget, IExternalLink, IInternalLink, \
    IInternalLinkCustomInfo, IInternalLinkCustomInfoTarget, IMailtoLink
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import ERROR_VALUE, IContentChecker
from pyams_content.features.json import IJSONConverter, JSONBaseConverter
from pyams_content.interfaces import IBaseContent
from pyams_content.reference.pictograms.interfaces import IPictogramTable
from pyams_i18n.interfaces import II18n
from pyams_sequence.interfaces import IInternalReference, ISequentialIdInfo
from pyams_sequence.reference import get_reference_target
from pyams_skin.layer import IPyAMSLayer, IPyAMSUserLayer
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.url import canonical_url, relative_url
from pyams_utils.vocabulary import vocabulary_config
from pyams_utils.zodb import volatile_property
from pyams_workflow.interfaces import IWorkflow, IWorkflowPublicationInfo


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Links vocabulary
#

@vocabulary_config(name='PyAMS content links')
class ContentLinksVocabulary(SimpleVocabulary):
    """Content links vocabulary"""

    def __init__(self, context=None):
        terms = []
        target = get_parent(context, IAssociationContainerTarget)
        if target is not None:
            terms = [SimpleTerm(link.__name__, title=IAssociationInfo(link).inner_title)
                     for link in IAssociationContainer(target).values() if
                     IBaseLink.providedBy(link)]
        super(ContentLinksVocabulary, self).__init__(terms)


#
# Base link persistent class
#

@implementer(IBaseLink)
class BaseLink(AssociationItem):
    """Base link persistent class"""

    title = FieldProperty(IBaseLink['title'])
    description = FieldProperty(IBaseLink['description'])
    _pictogram_name = FieldProperty(IBaseLink['pictogram_name'])

    @property
    def pictogram_name(self):
        return self._pictogram_name

    @pictogram_name.setter
    def pictogram_name(self, value):
        if value != self._pictogram_name:
            self._pictogram_name = value
            del self.pictogram

    @volatile_property
    def pictogram(self):
        table = query_utility(IPictogramTable)
        if table is not None:
            return table.get(self._pictogram_name)


class BaseLinkInfoAdapter(ContextAdapter):
    """Base link association info adapter"""

    @property
    def pictogram(self):
        return self.context.icon_class

    user_icon = None


class BaseLinkContentChecker(BaseContentChecker):
    """Base link content checker"""

    @property
    def label(self):
        request = check_request()
        translate = request.localizer.translate
        return II18n(self.context).query_attribute('title', request) or \
               '({0})'.format(translate(self.context.icon_hint).lower())


#
# Internal links
#

@implementer(IInternalReference)
class InternalReferenceMixin(object):
    """Internal reference mixin class"""

    reference = None

    @volatile_property
    def target(self):
        return get_reference_target(self.reference)

    def get_target(self, state=None, request=None):
        if request is None:
            request = check_request()
        if (not state) and not IPyAMSUserLayer.providedBy(request):
            return self.target
        return get_reference_target(self.reference, state, request)


@implementer(IInternalLink)
class InternalLink(BaseLink, InternalReferenceMixin):
    """Internal link persistent class"""

    icon_class = 'fa-external-link-square fa-rotate-90'
    icon_hint = _("Internal link")

    reference = FieldProperty(IInternalLink['reference'])
    force_canonical_url = FieldProperty(IInternalLink['force_canonical_url'])

    def is_visible(self, request=None):
        if not super().is_visible(request):
            return False
        target = self.get_target()
        if target is not None:
            publication_info = IWorkflowPublicationInfo(target, None)
            if publication_info is not None:
                return publication_info.is_visible(request)
        return False

    def get_editor_url(self):
        return 'oid://{0}'.format(self.reference)

    def get_url(self, request=None, view_name=None):
        target = self.get_target()
        if target is not None:
            if request is None:
                request = check_request()
            params = None
            if IInternalLinkCustomInfoTarget.providedBy(target):
                custom_info = IInternalLinkCustomInfo(self, None)
                if custom_info is not None:
                    params = custom_info.get_url_params()
                    if params:
                        params = urlencode(params)
            if self.force_canonical_url:
                return canonical_url(target, request, view_name, query=params)
            return relative_url(target, request, view_name=view_name, query=params)
        return ''


@subscriber(IObjectAddedEvent, context_selector=IInternalLink)
def handle_new_internal_link(event):
    """Check if link target is providing custom info"""
    link = event.object
    target = link.target
    if target is not None:
        info = IInternalLinkCustomInfoTarget(target, None)
        if info is not None:
            alsoProvides(link, info.internal_link_marker_interface)


@subscriber(IObjectModifiedEvent, context_selector=IInternalLink)
def handle_updated_internal_link(event):
    """Check when modified if new link target is providing custom info"""
    link = event.object
    # remove previous provided interfaces
    ifaces = tuple([iface for iface in directlyProvidedBy(link)
                    if issubclass(iface, IInternalLinkCustomInfo)])
    for iface in ifaces:
        noLongerProvides(link, iface)
    target = link.target
    if target is not None:
        info = IInternalLinkCustomInfoTarget(target, None)
        if info is not None:
            alsoProvides(link, info.internal_link_marker_interface)


@adapter_config(required=IInternalLink,
                provides=IAssociationInfo)
class InternalLinkAssociationInfoAdapter(BaseLinkInfoAdapter):
    """Internal link association info adapter"""

    @property
    def user_title(self):
        title = II18n(self.context).query_attribute('title')
        if not title:
            target = self.context.get_target()
            if target is not None:
                title = II18n(target).query_attribute('title')
        return title or '--'

    @property
    def user_header(self):
        description = II18n(self.context).query_attribute('description')
        if not description:
            target = self.context.get_target()
            if (target is not None) and hasattr(target, 'header'):
                description = II18n(target).query_attribute('header')
        return description

    @property
    def inner_title(self):
        target = self.context.get_target()
        if target is not None:
            sequence = ISequentialIdInfo(target)
            return '{0} ({1})'.format(II18n(target).query_attribute('title'),
                                      sequence.get_short_oid())
        return '--'

    @property
    def human_size(self):
        return '--'


@adapter_config(required=IInternalLink,
                provides=IContentChecker)
class InternalLinkContentChecker(BaseLinkContentChecker):
    """Internal link content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        content = get_parent(self.context, IBaseContent)
        if content is not None:
            workflow = IWorkflow(content, None)
            if workflow is not None:
                target = self.context.get_target(state=workflow.published_states)
                if target is None:
                    output.append(
                        translate(ERROR_VALUE).format(field=IInternalLink['reference'].title,
                                                      message=translate(
                                                          _("target is not published"))))
        return output


@adapter_config(required=(IInternalLink, IPyAMSLayer),
                provides=IJSONConverter)
class JSONInternalLinkConverter(JSONBaseConverter):
    """JSON internal link converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        target = self.context.target
        if target is None:
            return result
        result.update({
            'factory': 'link',
            'link_type': 'internal',
            'href': self.context.get_url(self.request)
        })
        title = II18n(self.context).query_attribute('title')
        if title:
            self.get_i18n_attribute(self.context, 'title', params.get('lang'), result)
        else:
            self.get_i18n_attribute(target, 'title', params.get('lang'), result)
        self.get_i18n_attribute(self.context, 'description', params.get('lang'), result)
        return result


#
# External links
#

@implementer(IExternalLink)
class ExternalLink(BaseLink):
    """External link persistent class"""

    icon_class = 'fa-link fa-rotate-90'
    icon_hint = _("External link")

    url = FieldProperty(IExternalLink['url'])
    language = FieldProperty(IExternalLink['language'])

    def get_editor_url(self):
        return self.url

    def get_url(self, request=None, view_name=None):
        return self.url


@adapter_config(required=IExternalLink,
                provides=IAssociationInfo)
class ExternalLinkAssociationInfoAdapter(BaseLinkInfoAdapter):
    """External link association info adapter"""

    @property
    def user_title(self):
        title = II18n(self.context).query_attribute('title')
        if not title:
            title = self.context.url
        return title or '--'

    @property
    def user_header(self):
        return II18n(self.context).query_attribute('description')

    @property
    def inner_title(self):
        return self.context.url

    @property
    def human_size(self):
        return '--'


@adapter_config(required=IExternalLink,
                provides=IContentChecker)
class ExternalLinkContentChecker(BaseLinkContentChecker):
    """External link content checker"""


@adapter_config(required=(IExternalLink, IPyAMSLayer),
                provides=IJSONConverter)
class JSONExternalLinkConverter(JSONBaseConverter):
    """JSON external link converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        url = self.context.url
        if not url:
            return result
        result.update({
            'factory': 'link',
            'link_type': 'external',
            'href': url
        })
        self.get_i18n_attribute(self.context, 'title', params.get('lang'), result)
        self.get_i18n_attribute(self.context, 'description', params.get('lang'), result)
        return result


#
# Mailto links
#

@implementer(IMailtoLink)
class MailtoLink(BaseLink):
    """Mailto link persistent class"""

    icon_class = 'fa-envelope-o'
    icon_hint = _("Mailto link")

    address = FieldProperty(IMailtoLink['address'])
    address_name = FieldProperty(IMailtoLink['address_name'])

    def get_editor_url(self):
        return 'mailto:{0} <{1}>'.format(self.address_name, self.address)

    def get_url(self, request=None, view_name=None):
        if self.address_name:
            return 'mailto:{}'.format(url_quote('{} <{}>'.format(self.address_name, self.address)))
        return 'mailto:{}'.format(self.address)


@adapter_config(required=IMailtoLink,
                provides=IAssociationInfo)
class MailtoLinkAssociationInfoAdapter(BaseLinkInfoAdapter):
    """Mailto link association info adapter"""

    @property
    def user_title(self):
        title = II18n(self.context).query_attribute('title')
        if not title:
            title = self.context.address_name
        return title or '--'

    @property
    def user_header(self):
        return II18n(self.context).query_attribute('description')

    @property
    def inner_title(self):
        if self.context.address_name:
            return escape('{} <{}>'.format(self.context.address_name, self.context.address))
        return self.context.address

    @property
    def human_size(self):
        return '--'


@adapter_config(required=IMailtoLink,
                provides=IContentChecker)
class MailtoLinkContentChecker(BaseLinkContentChecker):
    """Mailto link content checker"""


@adapter_config(required=(IMailtoLink, IPyAMSLayer),
                provides=IJSONConverter)
class JSONMailtoLinkConverter(JSONBaseConverter):
    """JSON mailto link converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        address = self.context.address
        if not address:
            return result
        result.update({
            'factory': 'link',
            'link_type': 'mailto',
            'address': address
        })
        self.get_i18n_attribute(self.context, 'title', params.get('lang'), result)
        self.get_i18n_attribute(self.context, 'description', params.get('lang'), result)
        self.get_attribute(self.context, 'address_name', result, name='name')
        return result
