#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from pyramid.decorator import reify
from pyramid.view import view_config
from z3c.form import field
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, implementer

from pyams_content.component.keynumber import KeyNumber
from pyams_content.component.keynumber.interfaces import IKeyNumber, IKeyNumberContainer, IKeyNumberContainerTarget, \
    KEYNUMBER_HIDDEN_FIELDS
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces.form import IInnerSubForm
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.column import I18nAttrColumn
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import switch_element_visibility
from pyams_skin.event import get_json_switched_table_refresh_event, get_json_table_row_refresh_event
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import BaseTable, I18nColumn, SorterColumn, TrashColumn, VisibilitySwitcherColumn
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from pyams_zmi.zmi.table import InnerTableView

__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Views marker interfaces
#

class IKeyNumbersView(Interface):
    """Key numbers view marker interface"""


class IKeyNumbersParentForm(Interface):
    """Key numbers parent form marker interface"""


#
# Key number items table view
#

class KeyNumbersTable(ProtectedFormObjectMixin, BaseTable):
    """Key numbers view inner table"""

    prefix = 'keynumbers'

    hide_header = True
    hide_body_toolbar = True
    sortOn = None

    @classmethod
    def get_context(cls, context):
        return IKeyNumberContainer(context)

    @reify
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        container = IKeyNumberContainer(self.context)
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-location': absolute_url(container, self.request),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-keynumbers-order.json'
        })
        attributes.setdefault('td', {}).update({
            'data-ams-attribute-switcher': self.get_switcher_target,
            'data-ams-switcher-attribute-name': self.get_switcher_attribute
        })
        return attributes

    @staticmethod
    def get_switcher_target(element, column):
        if column.__name__ == 'show-hide':
            return 'switch-keynumber-visibility.json'

    @staticmethod
    def get_switcher_attribute(element, column):
        if column.__name__ == 'show-hide':
            return 'visible'

    @reify
    def values(self):
        return list(super().values)


@adapter_config(context=(IKeyNumberContainer, IPyAMSLayer, KeyNumbersTable), provides=IValues)
class KeyNumbersTableValuesAdapter(ContextRequestViewAdapter):
    """Key numbers table values adapter"""

    @property
    def values(self):
        return IKeyNumberContainer(self.context).values()


@adapter_config(name='sorter', context=(IKeyNumberContainer, IPyAMSLayer, KeyNumbersTable), provides=IColumn)
class KeyNumbersTableSorterColumn(ProtectedFormObjectMixin, SorterColumn):
    """Key numbers table sorter column"""


@view_config(name='set-keynumbers-order.json', context=IKeyNumberContainer, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def set_keynumbers_order(request):
    """Update key numbers order"""
    order = list(map(str, json.loads(request.params.get('names'))))
    request.context.updateOrder(order)
    return {'status': 'success'}


@adapter_config(name='show-hide', context=(IKeyNumberContainer, IPyAMSLayer, KeyNumbersTable),
                provides=IColumn)
class KeyNumbersTableShowHideColumn(ProtectedFormObjectMixin, VisibilitySwitcherColumn):
    """Key numbers container visibility switcher column"""


@view_config(name='switch-keynumber-visibility.json', context=IKeyNumberContainer, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def switch_keynumber_visibility(request):
    """Switch key number visibility"""
    return switch_element_visibility(request, IKeyNumberContainer)


@adapter_config(name='label', context=(IKeyNumberContainer, IPyAMSLayer, KeyNumbersTable), provides=IColumn)
class KeyNumbersTableLabelColumn(I18nColumn, I18nAttrColumn):
    """Key numbers table label column"""

    _header = _('key-number-label', default="Header")
    attrName = 'label'
    weight = 10

    def getValue(self, obj):
        return super(KeyNumbersTableLabelColumn, self).getValue(obj) or '--'


@adapter_config(name='name', context=(IKeyNumberContainer, IPyAMSLayer, KeyNumbersTable), provides=IColumn)
class KeyNumbersTableNameColumn(I18nColumn, GetAttrColumn):
    """Key numbers table number column"""

    _header = _("Number")
    attrName = 'number'
    weight = 20

    def getValue(self, obj):
        return super(KeyNumbersTableNameColumn, self).getValue(obj) or '--'


@adapter_config(name='unit', context=(IKeyNumberContainer, IPyAMSLayer, KeyNumbersTable), provides=IColumn)
class KeyNumbersTableUnitColumn(I18nColumn, I18nAttrColumn):
    """Key numbers table unit column"""

    _header = _('key-number-unit', default="Unit")
    attrName = 'unit'
    weight = 30

    def getValue(self, obj):
        return super(KeyNumbersTableUnitColumn, self).getValue(obj) or '--'


@adapter_config(name='text', context=(IKeyNumberContainer, IPyAMSLayer, KeyNumbersTable), provides=IColumn)
class KeyNumbersTableTextColumn(I18nColumn, I18nAttrColumn):
    """Key numbers table text column"""

    _header = _("Associated text")
    attrName = 'text'
    weight = 40

    def getValue(self, obj):
        return super(KeyNumbersTableTextColumn, self).getValue(obj) or '--'


@adapter_config(name='trash', context=(IKeyNumberContainer, IPyAMSLayer, KeyNumbersTable), provides=IColumn)
class KeyNumbersTableTrashColumn(ProtectedFormObjectMixin, TrashColumn):
    """Key numbers table trash column"""


@adapter_config(name='keynumbers', context=(IKeyNumberContainerTarget, IPyAMSLayer, IKeyNumbersParentForm),
                provides=IInnerSubForm)
@implementer(IKeyNumbersView)
class KeyNumbersView(InnerTableView):
    """Key numbers view"""

    title = _("Key numbers")

    table_class = KeyNumbersTable
    weight = 110

    @property
    def actions_context(self):  # define context for internal actions
        return IKeyNumberContainer(self.context)


#
# Key numbers forms
#

@viewlet_config(name='add-keynumber.action', context=IKeyNumberContainerTarget, layer=IPyAMSLayer, view=IKeyNumbersView,
                manager=IWidgetTitleViewletManager, weight=1)
class KeyNumberAddAction(ProtectedFormObjectMixin, ToolbarAction):
    """Key number add action"""

    label = _("Add keynumber")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-keynumber.html'
    modal_target = True


@pagelet_config(name='add-keynumber.html', context=IKeyNumberContainer, layer=IPyAMSLayer)
@ajax_config(name='add-keynumber.json', context=IKeyNumberContainer, layer=IPyAMSLayer,
             base=AJAXAddForm)
class KeyNumberAddForm(AdminDialogAddForm):
    """Key number add form"""

    legend = _("Add new keynumber")
    icon_css_class = 'fa fa-fw fa-dashboard'

    fields = field.Fields(IKeyNumber).omit(*KEYNUMBER_HIDDEN_FIELDS)
    edit_permission = None  # use context permission checker

    def create(self, data):
        return KeyNumber()

    def add(self, object):
        IKeyNumberContainer(self.context).append(object)

    def get_ajax_output(self, changes):
        return {
            'status': 'success',
            'message': self.request.localizer.translate(_("Key number was correctly added")),
            'events': [
                get_json_switched_table_refresh_event(self.context, self.request, KeyNumbersTable)
            ]
        }


@pagelet_config(name='properties.html', context=IKeyNumber, layer=IPyAMSLayer)
@ajax_config(name='properties.json', context=IKeyNumber, layer=IPyAMSLayer)
class KeyNumberPropertiesEditForm(AdminDialogEditForm):
    """Key number properties edit form"""

    prefix = 'keynumber_properties.'

    legend = _("Edit keynumber properties")
    icon_css_class = 'fa fa-fw fa-dashboard'

    fields = field.Fields(IKeyNumber).omit(*KEYNUMBER_HIDDEN_FIELDS)
    edit_permission = None  # use context permission checker

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IKeyNumber, ())
        if updated:
            target = get_parent(self.context, IKeyNumberContainerTarget)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(target, self.request, KeyNumbersTable, self.context))
        return output
