#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from pyramid.events import subscriber
from zope.container.contained import Contained
from zope.container.ordered import OrderedContainer
from zope.interface import implementer
from zope.lifecycleevent import ObjectModifiedEvent
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectModifiedEvent, IObjectRemovedEvent
from zope.location.interfaces import ISublocations
from zope.location.location import locate
from zope.schema.fieldproperty import FieldProperty
from zope.traversing.interfaces import ITraversable

from pyams_catalog.utils import index_object
from pyams_content.component.keynumber.interfaces import IKeyNumber, IKeyNumberContainer, IKeyNumberContainerTarget, \
    KEYNUMBER_CONTAINER_KEY
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.json import IJSONConverter, JSONBaseConverter
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_current_registry, get_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent

from pyams_content import _


#
# Key number class and adapters
#

@implementer(IKeyNumber)
class KeyNumber(Persistent, Contained):
    """Key number persistent class"""

    visible = FieldProperty(IKeyNumber['visible'])
    label = FieldProperty(IKeyNumber['label'])
    number = FieldProperty(IKeyNumber['number'])
    unit = FieldProperty(IKeyNumber['unit'])
    text = FieldProperty(IKeyNumber['text'])


@adapter_config(required=IKeyNumber,
                provides=IFormContextPermissionChecker)
class KeyNumberPermissionChecker(ContextAdapter):
    """Key number permission checker"""

    @property
    def edit_permission(self):
        content = get_parent(self.context, IKeyNumberContainerTarget)
        return IFormContextPermissionChecker(content).edit_permission


@subscriber(IObjectAddedEvent, context_selector=IKeyNumber)
def handle_added_keynumber(event):
    """Handle added key number"""
    content = get_parent(event.object, IKeyNumberContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectModifiedEvent, context_selector=IKeyNumber)
def handle_modified_keynumber(event):
    """Handle modified key number"""
    content = get_parent(event.object, IKeyNumberContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectRemovedEvent, context_selector=IKeyNumber)
def handle_removed_keynumber(event):
    """Handle removed key number"""
    content = get_parent(event.object, IKeyNumberContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@adapter_config(required=IKeyNumber,
                provides=IContentChecker)
class KeyNumberContentChecker(BaseContentChecker):
    """Key number content checker"""

    @property
    def label(self):
        request = check_request()
        return II18n(self.context).query_attribute('title', request=request)

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language, )
        i18n = II18n(self.context)
        for lang in langs:
            for attr in ('label', 'text'):
                value = i18n.get_attribute(attr, lang, request)
                if not value:
                    field_title = translate(IKeyNumber[attr].title)
                    if len(langs) == 1:
                        output.append(translate(MISSING_VALUE).format(field=field_title))
                    else:
                        output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        return output


@adapter_config(required=(IKeyNumber, IPyAMSLayer),
                provides=IJSONConverter)
class JSONKeyNumberConverter(JSONBaseConverter):
    """JSON key number container"""

    def convert_content(self, params):
        result = super().convert_content(params)
        self.get_i18n_attribute(self.context, 'label', params.get('lang'), result)
        self.get_attribute(self.context, 'number', result)
        self.get_i18n_attribute(self.context, 'unit', params.get('lang'), result)
        self.get_i18n_attribute(self.context, 'text', params.get('lang'), result)
        return result


#
# Key numbers container classes and adapters
#

@factory_config(IKeyNumberContainer)
class KeyNumberContainer(OrderedContainer):
    """Key numbers container"""

    last_id = 1

    def append(self, value, notify=True):
        key = str(self.last_id)
        if not notify:
            # pre-locate key number item to avoid multiple notifications
            locate(value, self, key)
        self[key] = value
        self.last_id += 1
        if not notify:
            # make sure that key number item is correctly indexed
            index_object(value)

    def get_visible_items(self):
        yield from filter(lambda x: IKeyNumber(x).visible, self.values())


@adapter_config(required=IKeyNumberContainerTarget,
                provides=IKeyNumberContainer)
def keynumber_container_factory(target):
    """Key number container factory"""
    return get_annotation_adapter(target, KEYNUMBER_CONTAINER_KEY, IKeyNumberContainer,
                                  name='++keynumbers++')


@adapter_config(required=IKeyNumberContainer,
                provides=IFormContextPermissionChecker)
class KeyNumberContainerPermissionChecker(ContextAdapter):
    """Key number container permission checker"""

    @property
    def edit_permission(self):
        content = get_parent(self.context, IKeyNumberContainerTarget)
        return IFormContextPermissionChecker(content).edit_permission


@adapter_config(name='keynumbers',
                required=IKeyNumberContainerTarget,
                provides=ITraversable)
class KeyNumberContainerNamespace(ContextAdapter):
    """Key numbers container ++keynumbers++ namespace"""

    def traverse(self, name, furtherpath=None):
        return IKeyNumberContainer(self.context)


@adapter_config(name='keynumbers',
                required=IKeyNumberContainerTarget,
                provides=ISublocations)
class KeyNumberContainerSublocations(ContextAdapter):
    """Key numbers container sub-locations adapter"""

    def sublocations(self):
        return IKeyNumberContainer(self.context).values()


@adapter_config(name='keynumbers',
                required=IKeyNumberContainerTarget,
                provides=IContentChecker)
class KeyNumberContainerContentChecker(BaseContentChecker):
    """Key numbers container content checker"""

    label = _("Key numbers")
    sep = '\n'
    weight = 200

    def inner_check(self, request):
        output = []
        registry = request.registry
        for keynumber in IKeyNumberContainer(self.context).values():
            if not keynumber.visible:
                continue
            for name, checker in sorted(registry.getAdapters((keynumber, ), IContentChecker),
                                        key=lambda x: x[1].weight):
                output.append('- {0} ({1}):'.format(keynumber.number,
                                                    II18n(keynumber).query_attribute('label', request=request) or '--'))
                output.append(checker.get_check_output(request))
        return output
