#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyams_content.component.illustration import IIllustration, Illustration as BaseIllustration, \
    IllustrationContentChecker
from pyams_content.component.illustration.interfaces import IIllustrationParagraph, ILLUSTRATION_PARAGRAPH_NAME, \
    ILLUSTRATION_PARAGRAPH_TYPE
from pyams_content.component.paragraph import BaseParagraph, BaseParagraphFactory
from pyams_content.component.paragraph.api import JSONParagraphConverter
from pyams_content.component.paragraph.interfaces import IParagraphFactory
from pyams_content.features.checker.interfaces import IContentChecker
from pyams_content.features.json import IJSONConverter
from pyams_i18n.interfaces import II18n
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import utility_config
from pyams_utils.request import check_request


@factory_config(IIllustrationParagraph)
class Illustration(BaseIllustration, BaseParagraph):
    """Illustration class"""

    factory_name = ILLUSTRATION_PARAGRAPH_TYPE
    icon_class = 'fa-file-image-o'
    icon_hint = ILLUSTRATION_PARAGRAPH_NAME


@utility_config(name=ILLUSTRATION_PARAGRAPH_TYPE,
                provides=IParagraphFactory)
class IllustrationFactory(BaseParagraphFactory):
    """Illustration paragraph factory"""

    name = ILLUSTRATION_PARAGRAPH_NAME
    content_type = Illustration


@adapter_config(required=IIllustrationParagraph,
                provides=IContentChecker)
class IllustrationParagraphContentChecker(IllustrationContentChecker):
    """Illustration paragraph content checker"""

    @property
    def label(self):
        request = check_request()
        translate = request.localizer.translate
        return II18n(self.context).query_attribute('title', request) or \
            '({0})'.format(translate(self.context.icon_hint).lower())


@adapter_config(required=(IIllustrationParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONIllustrationParagraphRenderer(JSONParagraphConverter):
    """JSON illustration paragraph renderer"""

    def convert_content(self, params):
        result = super().convert_content(params)
        illustration = IIllustration(self.context, None)
        if not illustration:
            return result
        src = self.get_image_url(illustration, 'data', params)
        if not src:
            return result
        result.update({
            'src': src
        })
        self.get_i18n_attribute(illustration, 'title', params.get('lang'), result)
        self.get_i18n_attribute(illustration, 'alt_title', params.get('lang'), result)
        self.get_attribute(illustration, 'author', result)
        self.get_i18n_attribute(illustration, 'description', params.get('lang'), result)
        return result
