#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.decorator import reify
from pyramid.interfaces import IView
from pyramid.renderers import render
from pyramid.view import view_config
from transaction.interfaces import ITransactionManager
from z3c.form import button, field
from z3c.form.interfaces import INPUT_MODE
from zope.interface import Interface, implementer

from pyams_content.component.gallery import IBaseGallery
from pyams_content.component.gallery.interfaces import GALLERY_PARAGRAPH_TYPE, IGalleryParagraph
from pyams_content.component.gallery.zmi import GalleryMediasView
from pyams_content.component.gallery.zmi.file import GalleryMediaAddForm
from pyams_content.component.gallery.zmi.interfaces import IGalleryContentsView
from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
    BaseParagraphAddForm, BaseParagraphAddMenu, BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons, \
    ParagraphContainerTable, get_json_paragraph_markers_refresh_event, get_json_paragraph_refresh_event
from pyams_content.component.paragraph.zmi.container import ParagraphContainerCounterBase
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerTable, IParagraphContainerView, \
    IParagraphInnerEditor, IParagraphTitleToolbar
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.zmi import WfSharedContentPermissionMixin
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces.form import IInnerForm, IInnerSubForm
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import delete_container_element
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu, IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import get_table_id
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import ICacheKeyValue, VIEW_SYSTEM_PERMISSION
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import InnerAdminDisplayForm

from pyams_content import _


@viewlet_config(name='add-gallery.menu',
                context=IParagraphContainerTarget, view=IParagraphContainerView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=65)
class GalleryAddMenu(BaseParagraphAddMenu):
    """Gallery add menu"""

    label = _("Medias gallery...")
    label_css_class = 'fa fa-fw fa-picture-o'
    url = 'add-gallery.html'
    paragraph_type = GALLERY_PARAGRAPH_TYPE


@pagelet_config(name='add-gallery.html',
                context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-gallery.json',
             context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class GalleryAddForm(BaseParagraphAddForm):
    """Gallery add form"""

    legend = _("Add new gallery")

    content_interface = IGalleryParagraph


@pagelet_config(name='properties.html',
                context=IGalleryParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json',
             context=IGalleryParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class GalleryPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Gallery properties edit form"""

    prefix = 'gallery_properties.'

    legend = _("Edit gallery properties")

    content_interface = IGalleryParagraph


@adapter_config(context=(IGalleryParagraph, IPyAMSLayer),
                provides=IParagraphInnerEditor)
@implementer(IInnerForm)
class GalleryInnerEditForm(GalleryPropertiesEditForm):
    """Gallery properties inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()


#
# Paragraph container gallery marker
#

@viewlet_config(name='gallery',
                context=IBaseGallery, layer=IPyAMSLayer,
                view=IParagraphContainerTable, manager=IParagraphTitleToolbar,
                permission=VIEW_SYSTEM_PERMISSION, weight=5)
class ParagraphContainerGalleryMarker(ParagraphContainerCounterBase):
    """Paragraph container gallery marker column"""

    weight = 6
    action_class = 'action gallery-medias nowrap width-40'
    icon_class = 'fa fa-fw fa-picture-o'
    icon_hint = _("Visible medias")

    marker_type = 'gallery-medias'

    @property
    def count(self):
        return len(tuple(self.context.get_visible_medias()))


#
# Gallery contents view
#

@adapter_config(name='gallery-medias',
                context=(IGalleryParagraph, IPyAMSLayer, GalleryInnerEditForm),
                provides=IInnerSubForm)
@template_config(template='templates/gallery-medias.pt', layer=IPyAMSLayer)
@implementer(IGalleryContentsView)
class GalleryContentsView(WfSharedContentPermissionMixin, InnerAdminDisplayForm):
    """Gallery contents edit form"""

    _legend = _("Gallery medias")

    @property
    def legend(self):
        return self.request.localizer.translate(self._legend)

    fields = field.Fields(Interface)
    weight = 10

    @reify
    def gallery(self):
        return IBaseGallery(self.context)

    @property
    def gallery_name(self):
        return ICacheKeyValue(self.gallery)

    @property
    def medias(self):
        return self.gallery.values()

    def get_title(self, media):
        return II18n(media).query_attribute('title', request=self.request)

    def get_thumbnail_target(self, media):
        registry = self.request.registry
        view = registry.queryMultiAdapter((media, self.request), IView, name='preview.html')
        if view is not None:
            return absolute_url(media, self.request, 'preview.html')


@viewlet_config(name='add-media.menu',
                context=IGalleryParagraph, view=GalleryContentsView,
                manager=IWidgetTitleViewletManager)
class GalleryParagraphMediaAddMenu(WfSharedContentPermissionMixin, ToolbarAction):
    """Gallery media add menu"""

    label = _("Add media(s)")

    url = 'add-media.html'
    modal_target = True
    stop_propagation = True


@pagelet_config(name='add-media.html',
                context=IGalleryParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-media.json',
             context=IGalleryParagraph, layer=IPyAMSLayer, base=AJAXAddForm)
class GalleryParagraphMediaAddForm(GalleryMediaAddForm):
    """Gallery media add form"""

    def get_ajax_output(self, changes):
        ITransactionManager(self.context).commit()
        medias = []
        view = GalleryMediasView(self.context, self.request)
        for media in changes:
            medias.append(render('templates/gallery-media.pt',
                                 {'media': media, 'view': view}, request=self.request))
        output = {
            'status': 'success',
            'message': self.request.localizer.translate(_("Media(s) successfully added")),
            'events': [
                get_json_paragraph_markers_refresh_event(self.context, self.request,
                                                         self, ParagraphContainerGalleryMarker),
                get_json_paragraph_refresh_event(self.context, self.request)
            ],
            'callback': 'PyAMS_content.galleries.addMediaCallback',
            'options': {
                'parent': '{0}::{1}'.format(get_table_id(ParagraphContainerTable,
                                                         context=get_parent(self.context, IParagraphContainerTarget)),
                                            ICacheKeyValue(self.context)),
                'medias': medias
            }
        }
        return output


@view_config(name='delete-element.json',
             context=IGalleryParagraph, request_type=IPyAMSLayer,
             renderer='json', xhr=True)
def delete_media(request):
    """Delete media from container"""
    result = delete_container_element(request, IGalleryParagraph)
    if result.get('status') == 'success':
        form = GalleryInnerEditForm(request.context, request)
        form.update()
        result.update({
            'handle_json': True,
            'events': [
                get_json_paragraph_markers_refresh_event(request.context, request,
                                                         form, ParagraphContainerGalleryMarker),
                get_json_paragraph_refresh_event(request.context, request)
            ]
        })
    return result
