#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json
import zipfile
from io import BytesIO

from pyramid.decorator import reify
from pyramid.exceptions import Forbidden, NotFound
from pyramid.interfaces import IView
from pyramid.renderers import render_to_response
from pyramid.response import Response
from pyramid.view import view_config
from z3c.form import button, field
from zope.interface import Interface, implementer

from pyams_content.component.gallery import IGalleryTarget
from pyams_content.component.gallery.interfaces import IBaseGallery, IGallery
from pyams_content.component.gallery.zmi.interfaces import IGalleryContentsView
from pyams_content.component.paragraph import IBaseParagraph
from pyams_content.component.paragraph.zmi import get_json_paragraph_markers_refresh_event
from pyams_content.features.renderer.zmi.widget import RendererFieldWidget
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION, MANAGE_SITE_ROOT_PERMISSION
from pyams_content.root import ISiteRoot
from pyams_content.shared.common.zmi import WfSharedContentPermissionMixin
from pyams_file.interfaces import IFileInfo
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IWidgetForm, IWidgetsPrefixViewletsManager
from pyams_form.security import get_edit_permission
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.container import delete_container_element
from pyams_skin.interfaces import IInnerPage
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuItem
from pyams_template.template import template_config
from pyams_utils.interfaces import ICacheKeyValue, VIEW_SYSTEM_PERMISSION
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from pyams_zmi.form import AdminDialogDisplayForm, AdminDialogEditForm, AdminEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu

__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Gallery properties
#

@viewlet_config(name='gallery.menu',
                context=IGalleryTarget, layer=IPyAMSLayer,
                manager=IPropertiesMenu, weight=15,
                permission=VIEW_SYSTEM_PERMISSION)
class GalleryMenu(MenuItem):
    """Gallery menu item"""

    label = _("Medias gallery...")
    icon_class = 'fa-picture-o'
    url = '#gallery.html'


@viewlet_config(name='gallery.menu',
                context=ISiteRoot, layer=IPyAMSLayer,
                manager=IPropertiesMenu, weight=15,
                permission=MANAGE_SITE_ROOT_PERMISSION)
class SiteRootGalleryMenu(GalleryMenu):
    """Site root gallery menu"""


@pagelet_config(name='gallery.html',
                context=IGalleryTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IGalleryContentsView, IWidgetForm, IInnerPage)
class GalleryView(AdminEditForm):
    """Gallery view"""

    legend = _("Update gallery contents")
    fields = field.Fields(Interface)
    buttons = button.Buttons(Interface)


@pagelet_config(name='gallery.html',
                context=ISiteRoot, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
class SiteRootGalleryView(GalleryView):
    """Site root gallery view"""


@pagelet_config(name='properties.html',
                context=IGallery, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json',
             context=IGallery, layer=IPyAMSLayer)
class GalleryPropertiesEditForm(AdminDialogEditForm):
    """Gallery properties edit form"""

    prefix = 'gallery_properties.'

    legend = _("Update gallery properties")
    icon_css_class = 'fa fa-fw fa-picture-o'

    fields = field.Fields(IGallery).omit('__parent__', '__file__')
    fields['renderer'].widgetFactory = RendererFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def get_ajax_output(self, changes):
        if 'title' in changes.get(IGallery, ()):
            return {
                'status': 'reload',
                'location': '#external-files.html'
            }
        return super(self.__class__, self).get_ajax_output(changes)


#
# Gallery contents
#

@pagelet_config(name='contents.html',
                context=IBaseGallery, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IGalleryContentsView)
class GalleryContentForm(AdminDialogDisplayForm):
    """Gallery contents form"""

    legend = _("Update gallery contents")
    dialog_class = 'modal-max'

    fields = field.Fields(Interface)
    show_widget_title = True


class BaseGalleryMediasViewlet(Viewlet):
    """Base gallery medias viewlet"""

    permission = MANAGE_CONTENT_PERMISSION

    @reify
    def gallery(self):
        return IBaseGallery(self.context)

    @property
    def gallery_name(self):
        return ICacheKeyValue(self.gallery)

    @property
    def medias(self):
        return self.gallery.values()

    def get_title(self, media):
        return II18n(media).query_attribute('title', request=self.request)


@viewlet_config(name='gallery-medias',
                context=IBaseGallery, view=IGalleryContentsView,
                manager=IWidgetsPrefixViewletsManager)
@template_config(template='templates/gallery-medias.pt', layer=IPyAMSLayer)
@implementer(IGalleryContentsView)
class GalleryMediasViewlet(BaseGalleryMediasViewlet):
    """Gallery medias viewlet"""


@viewlet_config(name='gallery-medias',
                context=IGalleryTarget, view=IGalleryContentsView,
                manager=IWidgetsPrefixViewletsManager)
@template_config(template='templates/gallery-target-medias.pt', layer=IPyAMSLayer)
@implementer(IGalleryContentsView)
class GalleryTargetMediasViewlet(BaseGalleryMediasViewlet):
    """Gallery target medias viewlet"""


@view_config(name='get-gallery-medias.html',
             context=IBaseGallery, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION)
@implementer(IGalleryContentsView)
class GalleryMediasView(WfSharedContentPermissionMixin):
    """Gallery medias view"""

    def __init__(self, context, request):
        self.context = context
        self.request = request

    def __call__(self):
        paragraph = get_parent(self.context, IBaseParagraph)
        if paragraph is None:
            return render_to_response('templates/gallery-target-medias.pt', {
                'view': self
            }, request=self.request)
        return render_to_response('templates/gallery-medias.pt', {
            'view': self
        }, request=self.request)

    @reify
    def gallery(self):
        return IBaseGallery(self.context)

    @property
    def gallery_name(self):
        return ICacheKeyValue(self.gallery)

    @property
    def medias(self):
        return self.gallery.values()

    def get_title(self, media):
        return II18n(media).query_attribute('title', request=self.request)

    def get_thumbnail_target(self, media):
        registry = self.request.registry
        view = registry.queryMultiAdapter((media, self.request), IView, name='preview.html')
        if view is not None:
            return absolute_url(media, self.request, 'preview.html')


@view_config(name='set-medias-order.json',
             context=IBaseGallery, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION, renderer='json', xhr=True)
@view_config(name='set-medias-order.json',
             context=IGalleryTarget, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION, renderer='json', xhr=True)
def set_medias_order(request):
    """Set gallery medias order"""
    gallery = IBaseGallery(request.context)
    permission = get_edit_permission(request, context=gallery)
    if not request.has_permission(permission, context=gallery):
        raise Forbidden()
    medias_names = json.loads(request.params.get('medias'))
    gallery.updateOrder(medias_names)
    return {'status': 'success'}


@view_config(name='set-media-visibility.json',
             context=IBaseGallery, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION, renderer='json', xhr=True)
@view_config(name='set-media-visibility.json',
             context=IGalleryTarget, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION, renderer='json', xhr=True)
def set_media_visibility(request):
    """Set gallery media visibility"""
    gallery = IBaseGallery(request.context)
    media = gallery.get(str(request.params.get('object_name')))
    if media is None:
        raise NotFound()
    permission = get_edit_permission(request, context=gallery)
    if not request.has_permission(permission, context=gallery):
        raise Forbidden()
    media.visible = not media.visible
    result = {'visible': media.visible}
    if IBaseParagraph.providedBy(gallery):
        from pyams_content.component.gallery.zmi.paragraph import \
            GalleryInnerEditForm, ParagraphContainerGalleryMarker
        form = GalleryInnerEditForm(gallery, request)
        form.update()
        result.update({
            'status': 'success',
            'handle_json': True,
            'events': [
                get_json_paragraph_markers_refresh_event(gallery, request, form, ParagraphContainerGalleryMarker)
            ]
        })
    return result


@view_config(name='delete-media.json',
             context=IBaseGallery, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION, renderer='json', xhr=True)
@view_config(name='delete-media.json',
             context=IGalleryTarget, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION, renderer='json', xhr=True)
def delete_media(request):
    """Delete media from gallery"""
    return delete_container_element(request, container_factory=IBaseGallery)


#
# Gallery medias downloader
#

@view_config(name='get-medias.zip',
             context=IGalleryTarget, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION)
@view_config(name='get-medias.zip',
             context=IBaseGallery, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION)
def get_medias_archive(request):
    """Get all gallery medias as ZIP file"""
    zip_data = BytesIO()
    zip_file = zipfile.ZipFile(zip_data, mode='w')
    for media in IBaseGallery(request.context).values():
        zip_file.writestr(IFileInfo(media.data).filename, media.data.data)
    zip_file.close()
    zip_data.seek(0)
    response = Response(content_type='application/zip',
                        content_disposition='attachment; filename="gallery.zip"')
    response.body_file = zip_data
    return response
