#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyams_content.component.gallery import BaseGallery
from pyams_content.component.gallery.interfaces import GALLERY_PARAGRAPH_NAME, GALLERY_PARAGRAPH_TYPE, IGalleryParagraph
from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, BaseParagraphFactory
from pyams_content.component.paragraph.interfaces import IParagraphFactory
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, MISSING_VALUE
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.traversing import get_parent


@factory_config(IGalleryParagraph)
class Gallery(BaseGallery, BaseParagraph):
    """Gallery class"""

    factory_name = GALLERY_PARAGRAPH_TYPE
    icon_class = 'fa-picture-o'
    icon_hint = GALLERY_PARAGRAPH_NAME


@utility_config(name=GALLERY_PARAGRAPH_TYPE,
                provides=IParagraphFactory)
class GalleryFactory(BaseParagraphFactory):
    """Gallery paragraph factory"""

    name = GALLERY_PARAGRAPH_NAME
    content_type = Gallery


@adapter_config(required=IGalleryParagraph,
                provides=IContentChecker)
class GalleryParagraphContentChecker(BaseParagraphContentChecker):
    """Gallery paragraph content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('title', lang, request)
            if not value:
                field_title = translate(IGalleryParagraph['title'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        return output
