#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.events import subscriber
from pyramid.threadlocal import get_current_registry
from zope.lifecycleevent import ObjectModifiedEvent
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectModifiedEvent, IObjectRemovedEvent
from zope.location import locate
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty
from zope.traversing.interfaces import ITraversable

from pyams_catalog.utils import index_object
from pyams_content.component.gallery.interfaces import GALLERY_CONTAINER_KEY, GALLERY_RENDERERS, IBaseGallery, IGallery, \
    IGalleryFile, IGalleryTarget
from pyams_content.component.illustration import IBasicIllustration, VirtualIllustration
from pyams_content.component.paragraph import IBaseParagraph
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import IContentChecker
from pyams_content.features.renderer import RenderedContentMixin, RenderersVocabulary
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_file.interfaces import IImage
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.container import BTreeOrderedContainer
from pyams_utils.factory import factory_config
from pyams_utils.list import boolean_iter
from pyams_utils.registry import get_global_registry
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config

from pyams_content import _


#
# Galleries container
#

@factory_config(provided=IBaseGallery)
class BaseGallery(RenderedContentMixin, BTreeOrderedContainer):
    """Base gallery persistent class"""

    renderer = FieldProperty(IBaseGallery['renderer'])

    last_id = 1

    def append(self, value, notify=True):
        key = str(self.last_id)
        if not notify:
            # pre-locate gallery item to avoid multiple notifications
            locate(value, self, key)
        self[key] = value
        self.last_id += 1
        if not notify:
            # make sure that gallery item is correctly indexed
            index_object(value)

    def get_visible_medias(self):
        yield from filter(lambda x: IGalleryFile(x).visible, self.values())

    def get_visible_images(self):
        yield from filter(lambda x: IImage.providedBy(x.data), self.get_visible_medias())


@factory_config(provided=IGallery)
class Gallery(BaseGallery):
    """Gallery persistent class"""

    title = FieldProperty(IGallery['title'])
    description = FieldProperty(IGallery['description'])


@adapter_config(context=IGalleryTarget, provides=IGallery)
def gallery_factory(target):
    """Galleries container factory"""
    return get_annotation_adapter(target, GALLERY_CONTAINER_KEY, IGallery, name='++gallery++')


@adapter_config(name='gallery', context=IGalleryTarget, provides=ITraversable)
class GalleryContainerNamespace(ContextAdapter):
    """++gallery++ namespace traverser"""

    def traverse(self, name, furtherpath=None):
        registry = get_global_registry()
        return registry.getAdapter(self.context, IGallery, name=name)


@adapter_config(name='gallery', context=IGalleryTarget, provides=ISublocations)
class GalleryContainerSublocations(ContextAdapter):
    """Galleries container sublocations"""

    def sublocations(self):
        return IGallery(self.context).values()


@adapter_config(context=IGallery, provides=IFormContextPermissionChecker)
class GalleryPermissionChecker(ContextAdapter):
    """Gallery permission checker"""

    @property
    def edit_permission(self):
        content = get_parent(self.context, IWfSharedContent)
        if content is not None:
            return IFormContextPermissionChecker(content).edit_permission
        return MANAGE_SITE_ROOT_PERMISSION


@subscriber(IObjectAddedEvent, context_selector=IGallery)
def handle_added_gallery(event):
    """Handle added gallery"""
    gallery = event.object
    if IBaseParagraph.providedBy(gallery):
        # there is another event subscriber for paragraphs,
        # so don't trigger event twice !
        return
    content = get_parent(gallery, IWfSharedContent)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectModifiedEvent, context_selector=IGallery)
def handle_modified_gallery(event):
    """Handle modified gallery"""
    gallery = event.object
    if IBaseParagraph.providedBy(gallery):
        # there is another event subscriber for paragraphs,
        # so don't trigger event twice !
        return
    content = get_parent(gallery, IWfSharedContent)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectRemovedEvent, context_selector=IGallery)
def handle_removed_gallery(event):
    """Handle removed gallery"""
    gallery = event.object
    if IBaseParagraph.providedBy(gallery):
        # there is another event subscriber for paragraphs,
        # so don't trigger event twice !
        return
    content = get_parent(gallery, IWfSharedContent)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@adapter_config(name='gallery', context=IGallery, provides=IContentChecker)
class GalleryContentChecker(BaseContentChecker):
    """Gallery content checker"""

    label = _("Gallery")
    sep = '\n'
    weight = 60

    def inner_check(self, request):
        output = []
        registry = request.registry
        for media in IGallery(self.context).values():
            if not media.visible:
                continue
            for name, checker in sorted(registry.getAdapters((media,), IContentChecker),
                                        key=lambda x: x[1].weight):
                output.append('- {0} : '.format(checker.label or
                                                II18n(media).query_attribute('title', request=request)))
                output.append(checker.get_check_output(request))
        return output


@adapter_config(name='gallery', context=IGalleryTarget, provides=IContentChecker)
def gallery_target_content_checker(context):
    gallery = IGallery(context)
    return IContentChecker(gallery, None)


@vocabulary_config(name=GALLERY_RENDERERS)
class GalleryRendererVocabulary(RenderersVocabulary):
    """Gallery renderers vocabulary"""

    content_interface = IBaseGallery


#
# Gallery illustration adapters
#

@adapter_config(context=IGalleryFile, provides=IBasicIllustration)
def gallery_file_illustration_adapter(file):
    """Gallery file illustration adapter"""
    illustration = VirtualIllustration(file.data)
    illustration.title = (file.title or {}).copy()
    illustration.alt_title = (file.alt_title or {}).copy()
    illustration.author = file.author
    return illustration


@adapter_config(context=IBaseGallery, provides=IBasicIllustration)
def gallery_illustration_adapter(gallery):
    """Gallery illustration adapter"""
    has_medias, medias = boolean_iter(gallery.get_visible_medias())
    if has_medias:
        return IBasicIllustration(next(medias), None)
