#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from z3c.form.browser.text import TextWidget
from z3c.form.interfaces import DISPLAY_MODE, INPUT_MODE, ITextWidget
from z3c.form.widget import FieldWidget
from zope.interface import implementer_only

from pyams_content.component.extfile.interfaces import IExtFileManagerInfo
from pyams_form.interfaces import IFormLayer
from pyams_form.widget import widgettemplate_config
from pyams_i18n.interfaces import II18n
from pyams_i18n.interfaces.widget import II18nTextLineWidget
from pyams_i18n.widget import I18nTextLineWidget


#
# External file title field widget
#

class IExtFileTitleWidget(ITextWidget):
    """External file title field widget interface"""


@widgettemplate_config(mode=INPUT_MODE, template='templates/extfile-title-input.pt', layer=IFormLayer)
@widgettemplate_config(mode=DISPLAY_MODE, template='templates/extfile-title-display.pt', layer=IFormLayer)
@implementer_only(IExtFileTitleWidget)
class ExtFileTitleWidget(TextWidget):
    """External file title widget"""

    @property
    def prefix(self):
        manager_info = IExtFileManagerInfo(self.request.root, None)
        if manager_info is not None:
            lang = getattr(self, 'lang', None)  # I18n widget
            if lang is not None:
                return (manager_info.default_title_prefix or {}).get(lang, '') or ''
            else:
                return II18n(manager_info).query_attribute('default_title_prefix', request=self.request) or ''
        else:
            return ''


def ExtFileTitleFieldWidget(field, request):
    """External file title widget factory"""
    return FieldWidget(field, ExtFileTitleWidget(request))


#
# I18n external file title field widget
#

class II18nExtFileTitleWidget(II18nTextLineWidget):
    """I18n external file title field widget interface"""


@implementer_only(II18nExtFileTitleWidget)
class I18nExtFileTitleWidget(I18nTextLineWidget):
    """I18n external file title widget"""


def I18nExtFileTitleFieldWidget(field, request):
    """I18n external file title widget factory"""
    widget = I18nExtFileTitleWidget(request)
    widget.widget_factory = ExtFileTitleFieldWidget
    return FieldWidget(field, widget)
