#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import os

from pyramid.events import subscriber
from pyramid.threadlocal import get_current_registry
from zope.interface import alsoProvides, implementer
from zope.lifecycleevent import ObjectModifiedEvent
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectModifiedEvent, \
    IObjectRemovedEvent
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_content.component.association import AssociationItem
from pyams_content.component.association.interfaces import IAssociationInfo
from pyams_content.component.extfile.interfaces import IBaseExtFile, IExtAudio, IExtFile, \
    IExtFileManagerInfo, IExtImage, IExtMedia, IExtVideo
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, \
    MISSING_VALUE
from pyams_content.features.json import IJSONConverter, JSONBaseConverter
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_file.file import EXTENSIONS_THUMBNAILS
from pyams_file.interfaces import IFileInfo, IImage, IResponsiveImage
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_i18n.property import I18nFileProperty
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import check_request
from pyams_utils.size import get_human_size
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


__docformat__ = 'restructuredtext'

from pyams_content import _


EXTERNAL_FILES_FACTORIES = {}


def register_file_factory(key, factory, name=None):
    """Register new file factory"""
    if key not in EXTERNAL_FILES_FACTORIES:
        EXTERNAL_FILES_FACTORIES[key] = (factory, name or key)


@vocabulary_config(name='PyAMS files factories')
class ExternalFilesFactoriesVocabulary(SimpleVocabulary):
    """External files factories vocabulary"""

    def __init__(self, context):
        terms = sorted([SimpleTerm(k, title=v[1]) for k, v in EXTERNAL_FILES_FACTORIES.items()],
                       key=lambda x: x.title)
        super(ExternalFilesFactoriesVocabulary, self).__init__(terms)


@implementer(IBaseExtFile)
class BaseExtFile(AssociationItem):
    """External file persistent class"""

    title = FieldProperty(IExtFile['title'])
    description = FieldProperty(IExtFile['description'])
    author = FieldProperty(IExtFile['author'])
    language = FieldProperty(IExtFile['language'])
    filename = FieldProperty(IExtFile['filename'])


@adapter_config(required=IBaseExtFile,
                provides=IAssociationInfo)
class BaseExtFileAssociationInfoAdapter(ContextAdapter):
    """Base external file association info adapter"""

    @property
    def pictogram(self):
        return self.context.icon_class

    @property
    def user_title(self):
        request = check_request()
        manager_info = IExtFileManagerInfo(request.root)
        title = II18n(self.context).query_attribute('title', request=request)
        if not title:
            title = self.context.filename
            if '.' in title:
                title, extension = title.rsplit('.', 1)
        return '{0} {1}'.format(II18n(manager_info).query_attribute('default_title_prefix',
                                                                    request=request) or '',
                                title)

    @property
    def user_header(self):
        request = check_request()
        return II18n(self.context).query_attribute('description', request=request)

    @property
    def user_icon(self):
        filename = self.context.filename
        if filename:
            name, ext = os.path.splitext(filename)
            return '<img class="margin-left-5 align-top" ' \
                'src="/--static--/myams/img/mimetypes/16x16/{thumb}" />'.format(
                 thumb=EXTENSIONS_THUMBNAILS.get(ext, 'unknown.png'))

    @property
    def inner_title(self):
        return self.context.filename or '--'

    @property
    def human_size(self):
        data = II18n(self.context).query_attribute('data')
        if data:
            return get_human_size(data.get_size())
        return '--'


def update_properties(extfile):
    """Update missing file properties"""
    request = check_request()
    i18n = query_utility(INegotiator)
    if i18n is not None:
        lang = i18n.server_language
        data = II18n(extfile).get_attribute('data', lang, request)
        if data:
            info = IFileInfo(data)
            info.title = II18n(extfile).get_attribute('title', lang, request)
            info.description = II18n(extfile).get_attribute('description', lang, request)
            if not extfile.filename:
                extfile.filename = info.filename
            else:
                info.filename = extfile.filename
            info.language = extfile.language


@subscriber(IObjectAddedEvent, context_selector=IBaseExtFile)
def handle_added_extfile(event):
    """Handle added external file"""
    # update inner file properties
    extfile = event.object
    update_properties(extfile)
    # notify content modification
    content = get_parent(extfile, IWfSharedContent)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectModifiedEvent, context_selector=IBaseExtFile)
def handle_modified_extfile(event):
    """Handle modified external file"""
    # update inner file properties
    extfile = event.object
    update_properties(extfile)
    # notify content modification
    content = get_parent(extfile, IWfSharedContent)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectRemovedEvent, context_selector=IBaseExtFile)
def handle_removed_extfile(event):
    """Handle removed external file"""
    content = get_parent(event.object, IWfSharedContent)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@implementer(IExtFile)
class ExtFile(BaseExtFile):
    """Generic external file persistent class"""

    icon_class = 'fa-file-text-o'
    icon_hint = _("Standard file")

    data = I18nFileProperty(IExtFile['data'])


register_file_factory('file', ExtFile, _("Standard file"))


@adapter_config(required=IExtFile,
                provides=IContentChecker)
class ExtFileContentChecker(BaseContentChecker):
    """Base external file content checker"""

    @property
    def label(self):
        request = check_request()
        translate = request.localizer.translate
        return II18n(self.context).query_attribute('title', request) or \
               '({0})'.format(translate(self.context.icon_hint).lower())

    interface = IExtFile

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for attr in ('title', 'data'):
            for lang in langs:
                value = i18n.get_attribute(attr, lang, request)
                if not value:
                    if len(langs) > 1:
                        output.append(translate(MISSING_LANG_VALUE).format(field=translate(self.interface[attr].title),
                                                                           lang=lang))
                    else:
                        output.append(translate(MISSING_VALUE).format(field=translate(self.interface[attr].title)))
        for attr in ('author', 'filename'):
            value = getattr(self.context, attr)
            if not value:
                output.append(translate(MISSING_VALUE).format(field=translate(self.interface[attr].title)))
        return output


@adapter_config(required=(IExtFile, IPyAMSLayer),
                provides=IJSONConverter)
class JSONExtFileConverter(JSONBaseConverter):
    """JSON external file converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        src = self.get_file_url(self.context, 'data', params)
        if src is None:
            return result
        result.update({
            'factory': 'file',
            'src': src,
            'content_type': II18n(self.context)
                .query_attribute('data', lang=params.get('lang')).content_type
        })
        data = II18n(self.context).query_attribute('data', request=self.request)
        if data:
            result['file_size'] = data.get_size()
        self.get_i18n_attribute(self.context, 'title', params.get('lang'), result)
        self.get_i18n_attribute(self.context, 'description', params.get('lang'), result)
        self.get_attribute(self.context, 'author', result)
        self.get_attribute(self.context, 'filename', result)
        return result


@implementer(IExtImage)
class ExtImage(BaseExtFile):
    """External image persistent class"""

    icon_class = 'fa-file-image-o'
    icon_hint = _("Image")

    _data = I18nFileProperty(IExtImage['data'])

    @property
    def data(self):
        return self._data

    @data.setter
    def data(self, value):
        self._data = value
        for data in self._data.values():
            if IImage.providedBy(data):
                alsoProvides(data, IResponsiveImage)
                
    @data.deleter
    def data(self):
        del self._data


register_file_factory('image', ExtImage, _("Image"))


@adapter_config(required=IExtImage,
                provides=IContentChecker)
class ExtImageContentChecker(ExtFileContentChecker):
    """External image content checker"""

    interface = IExtImage


@adapter_config(required=(IExtImage, IPyAMSLayer),
                provides=IJSONConverter)
class JSONExtImageConverter(JSONExtFileConverter):
    """JSON external image converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        if not result:
            return result
        src = self.get_image_url(self.context, 'data', params)
        if src:
            result.update({
                'factory': 'image',
                'src': src
            })
        return result


@implementer(IExtVideo)
class ExtVideo(BaseExtFile):
    """External video file persistent class"""

    icon_class = 'fa-file-video-o'
    icon_hint = _("Video")

    data = I18nFileProperty(IExtVideo['data'])


register_file_factory('video', ExtVideo, _("Video"))


@adapter_config(required=IExtVideo,
                provides=IContentChecker)
class ExtVideoContentChecker(ExtFileContentChecker):
    """External video file content checker"""

    interface = IExtVideo


@implementer(IExtAudio)
class ExtAudio(BaseExtFile):
    """External audio file persistent class"""

    icon_class = 'fa-file-audio-o'
    icon_hint = _("Audio file")

    data = I18nFileProperty(IExtAudio['data'])


register_file_factory('audio', ExtAudio, _("Audio file"))


@adapter_config(required=IExtAudio,
                provides=IContentChecker)
class ExtAudioContentChecker(ExtFileContentChecker):
    """External audio file content checker"""

    interface = IExtAudio
