#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import json

from pyramid.decorator import reify
from pyramid.view import view_config
from z3c.form import field
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, implementer

from pyams_content.component.association.interfaces import IAssociationContainer, \
    IAssociationContainerTarget, IAssociationInfo
from pyams_content.component.association.zmi.interfaces import IAssociationsParentForm, \
    IAssociationsView
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import AJAXAddForm, AJAXEditForm
from pyams_form.interfaces.form import IInnerSubForm
from pyams_form.security import ProtectedFormObjectMixin
from pyams_pagelet.pagelet import Pagelet, pagelet_config
from pyams_skin.container import delete_container_element, switch_element_visibility
from pyams_skin.event import get_json_switched_table_refresh_event, \
    get_json_table_row_refresh_event
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import BaseTable, I18nColumn, ImageColumn, NameColumn, SorterColumn, \
    TrashColumn, VisibilitySwitcherColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import get_current_registry
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogDisplayForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.view import ContainerAdminView
from pyams_zmi.zmi.table import InnerTableView


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Association item base forms
#

class AssociationItemAJAXAddForm(AJAXAddForm):
    """Association item add form, JSON renderer"""

    def get_ajax_output(self, changes):
        return {
            'status': 'success',
            'message': self.request.localizer.translate(_("Association was correctly added.")),
            'events': [
                get_json_switched_table_refresh_event(self.context, self.request, AssociationsTable)
            ]
        }


class AssociationItemAJAXEditForm(AJAXEditForm):
    """Association item properties edit form, JSON renderer"""

    def get_associations_table(self):
        target = get_parent(self.context, IAssociationContainerTarget)
        return {
            'status': 'success',
            'message': self.request.localizer.translate(self.successMessage),
            'events': [
                get_json_table_row_refresh_event(target, self.request, AssociationsTable, self.context)
            ]
        }


#
# Content associations view
#

@viewlet_config(name='associations.menu', context=IAssociationContainerTarget, layer=IPyAMSLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION, weight=200)
class AssociationsMenu(MenuItem):
    """Associations menu"""

    label = _("Associations...")
    icon_class = 'fa-link'
    url = '#associations.html'


class AssociationsTable(ProtectedFormObjectMixin, BaseTable):
    """Associations view inner table"""

    prefix = 'associations'
    associations_name = ''

    hide_header = True
    sortOn = None

    @property
    def cssClasses(self):
        classes = ['table', 'table-bordered', 'table-striped', 'table-hover', 'table-tight']
        permission = self.permission
        if (not permission) or self.request.has_permission(permission, context=self.context):
            classes.append('table-dnd')
        return {'table': ' '.join(classes)}

    @reify
    def data_attributes(self):
        registry = get_current_registry()
        target = get_parent(self.context, IAssociationContainerTarget)
        container = registry.getAdapter(target, IAssociationContainer, name=self.associations_name)
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-location': absolute_url(container, self.request),
            'data-ams-tablednd-drag-handle': 'td.sorter',
            'data-ams-tablednd-drop-target': 'set-associations-order.json'
        })
        attributes.setdefault('td', {}).update({
            'data-ams-attribute-switcher': self.get_switcher_target
        })
        return attributes

    @staticmethod
    def get_switcher_target(element, column):
        if column.__name__ == 'show-hide':
            return 'switch-association-visibility.json'

    @reify
    def values(self):
        return list(super().values)


@adapter_config(context=(IAssociationContainerTarget, IPyAMSLayer, AssociationsTable), provides=IValues)
class AssociationsTableValuesAdapter(ContextRequestViewAdapter):
    """Associations table values adapter"""

    @property
    def values(self):
        registry = get_current_registry()
        return registry.getAdapter(self.context, IAssociationContainer, name=self.view.associations_name).values()


@adapter_config(name='sorter', context=(IAssociationContainerTarget, IPyAMSLayer, AssociationsTable), provides=IColumn)
@adapter_config(name='sorter', context=(IAssociationContainer, IPyAMSLayer, AssociationsTable), provides=IColumn)
class AssociationsTableSorterColumn(ProtectedFormObjectMixin, SorterColumn):
    """Associations table sorter column"""


@view_config(name='set-associations-order.json', context=IAssociationContainer, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def set_associations_order(request):
    """Update associations order"""
    order = list(map(str, json.loads(request.params.get('names'))))
    request.context.updateOrder(order)
    return {'status': 'success'}


@adapter_config(name='show-hide', context=(IAssociationContainerTarget, IPyAMSLayer, AssociationsTable),
                provides=IColumn)
@adapter_config(name='show-hide', context=(IAssociationContainer, IPyAMSLayer, AssociationsTable),
                provides=IColumn)
class AssociationsTableShowHideColumn(ProtectedFormObjectMixin, VisibilitySwitcherColumn):
    """Associations container visibility switcher column"""


@view_config(name='switch-association-visibility.json', context=IAssociationContainer, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def set_association_visibility(request):
    """Set association visibility"""
    return switch_element_visibility(request, IAssociationContainer)


@adapter_config(name='pictogram', context=(IAssociationContainerTarget, IPyAMSLayer, AssociationsTable), 
                provides=IColumn)
@adapter_config(name='pictogram', context=(IAssociationContainer, IPyAMSLayer, AssociationsTable), 
                provides=IColumn)
class AssociationsTablePictogramColumn(ImageColumn):
    """Associations table pictogram column"""

    weight = 8

    def get_icon_class(self, item):
        info = IAssociationInfo(item, None)
        if info is not None:
            return info.pictogram

    def get_icon_hint(self, item):
        return self.request.localizer.translate(item.icon_hint)


@adapter_config(name='name', context=(IAssociationContainerTarget, IPyAMSLayer, AssociationsTable), 
                provides=IColumn)
@adapter_config(name='name', context=(IAssociationContainer, IPyAMSLayer, AssociationsTable), 
                provides=IColumn)
class AssociationsTablePublicNameColumn(NameColumn):
    """Associations table name column"""

    _header = _("Public title")

    def getValue(self, obj):
        info = IAssociationInfo(obj, None)
        if info is not None:
            title = info.user_title
        else:
            title = '--'
        if info.user_icon:
            title += ' ' + info.user_icon
        return title


@adapter_config(name='inner_name', context=(IAssociationContainerTarget, IPyAMSLayer, AssociationsTable), 
                provides=IColumn)
@adapter_config(name='inner_name', context=(IAssociationContainer, IPyAMSLayer, AssociationsTable), 
                provides=IColumn)
class AssociationsTableInnerNameColumn(I18nColumn, GetAttrColumn):
    """Associations table inner name column"""

    _header = _("Inner title")
    weight = 20

    def getValue(self, obj):
        info = IAssociationInfo(obj, None)
        if info is not None:
            return info.inner_title
        else:
            return '--'


@adapter_config(name='size', context=(IAssociationContainerTarget, IPyAMSLayer, AssociationsTable), 
                provides=IColumn)
@adapter_config(name='size', context=(IAssociationContainer, IPyAMSLayer, AssociationsTable), 
                provides=IColumn)
class AssociationsTableSizeColumn(I18nColumn, GetAttrColumn):
    """Associations table size column"""

    _header = _("Size")
    weight = 30

    def getValue(self, obj):
        info = IAssociationInfo(obj, None)
        if info is not None:
            return info.human_size
        else:
            return '--'


@adapter_config(name='trash', context=(IAssociationContainerTarget, IPyAMSLayer, AssociationsTable),
                provides=IColumn)
@adapter_config(name='trash', context=(IAssociationContainer, IPyAMSLayer, AssociationsTable), 
                provides=IColumn)
class AssociationsTableTrashColumn(ProtectedFormObjectMixin, TrashColumn):
    """Associations table trash column"""


@view_config(name='delete-element.json', context=IAssociationContainer, request_type=IPyAMSLayer,
             renderer='json', xhr=True)
def delete_association(request):
    """Delete association"""
    output = delete_container_element(request)
    if output.get('status') == 'success':
        from pyams_content.component.paragraph.zmi import get_json_paragraph_toolbar_refresh_event
        parent = get_parent(request.context, IAssociationContainerTarget)
        output.update({
            'handle_json': True,
            'events': [
                get_json_paragraph_toolbar_refresh_event(parent, request),
                get_json_switched_table_refresh_event(parent, request, AssociationsTable)
            ]
        })
    return output


@pagelet_config(name='associations.html', context=IAssociationContainerTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IAssociationsView)
class AssociationsContainerView(ContainerAdminView, Pagelet):
    """Associations container view"""

    title = _("Associations list")
    table_class = AssociationsTable


@pagelet_config(name='associations-dialog.html', context=IAssociationContainerTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@implementer(IAssociationsParentForm)
class AssociationsContainerDialogView(AdminDialogDisplayForm):
    """Associations dialog view"""

    title = _("Associations list")
    dialog_class = 'modal-large no-widget-toolbar'
    fieldset_class = 'height-300'

    fields = field.Fields(Interface)


@adapter_config(name='associations', context=(IAssociationContainerTarget, IPyAMSLayer, IAssociationsParentForm),
                provides=IInnerSubForm)
@implementer(IAssociationsView)
class AssociationsView(InnerTableView):
    """Associations view"""

    title = _("Associations")

    table_class = AssociationsTable
    weight = 90
