#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.association.interfaces.paragraph import ASSOCIATION_PARAGRAPH_NAME, \
    ASSOCIATION_PARAGRAPH_RENDERERS, ASSOCIATION_PARAGRAPH_TYPE, IAssociationParagraph
from pyams_content.component.extfile.interfaces import IExtFileContainerTarget
from pyams_content.component.links.interfaces import ILinkContainerTarget
from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, BaseParagraphFactory
from pyams_content.component.paragraph.interfaces import IParagraphFactory
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.renderer import RenderersVocabulary
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


@factory_config(provided=IAssociationParagraph)
@implementer(IExtFileContainerTarget, ILinkContainerTarget)
class AssociationParagraph(BaseParagraph):
    """Associations paragraph"""

    factory_name = ASSOCIATION_PARAGRAPH_TYPE
    icon_class = 'fa-link'
    icon_hint = ASSOCIATION_PARAGRAPH_NAME

    renderer = FieldProperty(IAssociationParagraph['renderer'])


@utility_config(name=ASSOCIATION_PARAGRAPH_TYPE,
                provides=IParagraphFactory)
class AssociationParagraphFactory(BaseParagraphFactory):
    """Associations paragraph factory"""

    name = ASSOCIATION_PARAGRAPH_NAME
    content_type = AssociationParagraph


@adapter_config(required=IAssociationParagraph,
                provides=IContentChecker)
class AssociationParagraphContentChecker(BaseParagraphContentChecker):
    """Associations paragraph content checker"""

    @property
    def label(self):
        request = check_request()
        translate = request.localizer.translate
        return II18n(self.context).query_attribute('title', request) or \
            '({0})'.format(translate(self.context.icon_hint).lower())

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language, )
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('title', lang, request)
            if not value:
                field_title = translate(IAssociationParagraph['title'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        return output


@vocabulary_config(name=ASSOCIATION_PARAGRAPH_RENDERERS)
class AssociationParagraphRendererVocabulary(RenderersVocabulary):
    """Associations paragraph renderers vocabulary"""

    content_interface = IAssociationParagraph
