#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.container.ordered import OrderedContainer
from zope.location import locate
from zope.location.interfaces import ISublocations
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary
from zope.traversing.interfaces import ITraversable

from pyams_catalog.utils import index_object
from pyams_content.component.association.interfaces import ASSOCIATION_CONTAINER_KEY, \
    IAssociationContainer, IAssociationContainerTarget, IAssociationInfo, IAssociationItem
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import IContentChecker
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_current_registry
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


@factory_config(IAssociationContainer)
class AssociationContainer(OrderedContainer):
    """Associations container"""

    last_id = 1

    def append(self, value, notify=True):
        key = str(self.last_id)
        if not notify:
            # pre-locate association item to avoid multiple notifications
            locate(value, self, key)
        self[key] = value
        self.last_id += 1
        if not notify:
            # make sure that association item is correctly indexed
            index_object(value)

    def get_visible_items(self, request=None):
        for item in filter(lambda x: IAssociationItem(x).visible, self.values()):
            if IAdminLayer.providedBy(request) or item.is_visible(request):
                yield item


@adapter_config(context=IAssociationContainerTarget, provides=IAssociationContainer)
def association_container_factory(target):
    """Associations container factory"""
    return get_annotation_adapter(target, ASSOCIATION_CONTAINER_KEY, IAssociationContainer,
                                  name='++ass++')


@adapter_config(name='ass', context=IAssociationContainerTarget, provides=ITraversable)
class AssociationContainerNamespace(ContextAdapter):
    """Associations container ++ass++ namespace"""

    def traverse(self, name, furtherpath=None):
        registry = get_current_registry()
        return registry.queryAdapter(self.context, IAssociationContainer, name=name or '')


@adapter_config(name='associations', context=IAssociationContainerTarget, provides=ISublocations)
class AssociationContainerSublocations(ContextAdapter):
    """Associations container sub-locations adapter"""

    def sublocations(self):
        return IAssociationContainer(self.context).values()


@adapter_config(name='associations', context=IAssociationContainerTarget, provides=IContentChecker)
class AssociationsContentChecker(BaseContentChecker):
    """Associations content checker"""

    label = _("Associations")
    sep = '\n'
    weight = 100

    def inner_check(self, request):
        output = []
        registry = request.registry
        for association in IAssociationContainer(self.context).values():
            if not association.visible:
                continue
            info = IAssociationInfo(association)
            for name, checker in sorted(registry.getAdapters((association,), IContentChecker),
                                        key=lambda x: x[1].weight):
                output.append('- {0} :'.format(info.user_title or
                                               '({0})'.format(info.inner_title)))
                output.append(checker.get_check_output(request))
        return output


@vocabulary_config(name='PyAMS content associations')
class ContentAssociationsVocabulary(SimpleVocabulary):
    """Content associations vocabulary"""

    def __init__(self, context=None):
        terms = []
        target = get_parent(context, IAssociationContainerTarget)
        if target is not None:
            terms = [SimpleTerm(link.__name__, title=IAssociationInfo(link).inner_title)
                     for link in IAssociationContainer(target).values()]
        super(ContentAssociationsVocabulary, self).__init__(terms)
