#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from pyams_content.interfaces import MANAGER_ROLE, MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.interfaces import IManagerRestrictions
from pyams_i18n.interfaces import II18n
from pyams_notify.interfaces import INotification, INotificationHandler
from pyams_security.interfaces import IProtectedObject
from pyams_workflow.interfaces import IWorkflowTransitionEvent

# import packages
from pyams_notify.event import Notification
from pyams_utils.adapter import adapter_config, ContextAdapter
from pyams_utils.request import query_request
from pyams_utils.url import absolute_url
from pyramid.events import subscriber
from pyramid.location import lineage


@subscriber(IWorkflowTransitionEvent)
def handle_workflow_event(event):
    """Handle workflow transition event"""
    request = query_request()
    if request is None:
        return
    transition = event.transition
    if transition.user_data.get('notify_roles') is None:
        return
    translate = request.localizer.translate
    notification = Notification(request=request,
                                context=event.object,
                                action='notify',
                                category='content.workflow',
                                message=translate(transition.user_data['notify_message']).format(
                                    II18n(event.object).query_attribute('title', request=request)),
                                url=absolute_url(event.object, request, 'admin'),
                                transition=transition)
    notification.send()


@adapter_config(name='content.workflow', context=INotification, provides=INotificationHandler)
class ContentWorkflowTransitionNotificationHandler(ContextAdapter):
    """Content workflow transition notification handler"""

    def get_target(self):
        notified_roles = self.context.user_data['transition'].user_data['notify_roles']
        if '*' in notified_roles:
            return {}
        notification_source = self.context.context
        principals = set()
        for context in lineage(notification_source):
            protection = IProtectedObject(context, None)
            if protection is not None:
                for role_id in notified_roles:
                    for principal in protection.get_principals(role_id):
                        if role_id == MANAGER_ROLE:
                            if principal in notification_source.managers:
                                principals.add(principal)
                            else:
                                restrictions = IManagerRestrictions(notification_source, None)
                                if restrictions is not None:
                                    principal_restrictions = restrictions.get_restrictions(
                                        principal)
                                    if principal_restrictions and \
                                            principal_restrictions.check_access(
                                                notification_source,
                                                MANAGE_CONTENT_PERMISSION):
                                        principals.add(principal)
                        else:
                            principals.add(principal)
        source_id = self.context.source['id']
        if source_id in principals:
            principals.remove(source_id)
        return {'principals': tuple(principals)}
