#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.interface import implementer

from pyams_content.interfaces import CREATE_VERSION_PERMISSION, MANAGER_ROLE, MANAGE_CONTENT_PERMISSION, \
    MANAGE_SITE_ROOT_PERMISSION, OWNER_ROLE, PILOT_ROLE, PUBLISH_CONTENT_PERMISSION, WEBMASTER_ROLE
from pyams_content.shared.common.interfaces import IContributorRestrictions, IManagerRestrictions
from pyams_content.workflow.basic import ARCHIVED, ARCHIVED_STATES, BasicWorkflow, DRAFT, MANAGER_STATES, \
    PRE_PUBLISHED, PROTECTED_STATES, PUBLISHED, PUBLISHED_STATES, READONLY_STATES, RETIRED_STATES, \
    STATES_VOCABULARY, UPDATE_STATES, VISIBLE_STATES, WAITING_STATES, archive_action, archived_to_draft, \
    cancel_prepublish_action, clone_action, delete, draft_to_prepublished, init, \
    prepublished_to_published, publish_action
from pyams_content.workflow.interfaces import IAlertsWorkflow
from pyams_security.interfaces import INTERNAL_USER_ID
from pyams_utils.registry import utility_config
from pyams_utils.request import check_request
from pyams_workflow.interfaces import IWorkflow, IWorkflowVersions
from pyams_workflow.workflow import Transition

__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Workflow conditions
#

def can_manage_content(wf, context):
    """Check if a manager can handle content"""
    request = check_request()
    # grant access to webmaster
    if request.has_permission(MANAGE_SITE_ROOT_PERMISSION, context=context):
        return True
    # owner and content managers can manage content
    principal_id = request.principal.id
    if principal_id in {INTERNAL_USER_ID} | context.owner | context.managers:
        return True
    # shared tool managers can manage content if restrictions apply
    restrictions = IManagerRestrictions(context).get_restrictions(principal_id)
    if restrictions and restrictions.check_access(context,
                                                  permission=PUBLISH_CONTENT_PERMISSION,
                                                  request=request):
        return True
    return request.has_permission(MANAGE_CONTENT_PERMISSION, context=context)


def can_create_new_version(wf, context):
    """Check if we can create a new version"""
    # can't create new version when previous draft already exists
    versions = IWorkflowVersions(context)
    if versions.has_version(DRAFT) or versions.has_version(PRE_PUBLISHED):
        return False
    request = check_request()
    # grant access to webmaster
    if request.has_permission(MANAGE_SITE_ROOT_PERMISSION, context=context):
        return True
    # grant access to owner, creator and local contributors
    principal_id = request.principal.id
    if principal_id in context.owner | {context.creator} | context.contributors:
        return True
    # grant access to allowed contributors
    restrictions = IContributorRestrictions(context).get_restrictions(principal_id)
    if restrictions and restrictions.check_access(context,
                                                  permission=MANAGE_CONTENT_PERMISSION,
                                                  request=request):
        return True
    # grant access to local content managers
    if principal_id in context.managers:
        return True
    # grant access to shared tool managers if restrictions apply
    restrictions = IManagerRestrictions(context).get_restrictions(principal_id)
    if restrictions and restrictions.check_access(context,
                                                  permission=CREATE_VERSION_PERMISSION,
                                                  request=request):
        return True
    return request.has_permission(CREATE_VERSION_PERMISSION, context=context)


def can_archive_content(wf, context):
    """Check if content can be archived"""
    if can_manage_content(wf, context):
        return True
    request = check_request()
    return request.has_permission(MANAGE_CONTENT_PERMISSION, context=context)


#
# Workflow transitions
#

prepublished_to_draft = Transition(transition_id='prepublished_to_draft',
                                   title=_("Cancel publication"),
                                   source=PRE_PUBLISHED,
                                   destination=DRAFT,
                                   permission=MANAGE_CONTENT_PERMISSION,
                                   condition=can_manage_content,
                                   action=cancel_prepublish_action,
                                   menu_css_class='fa fa-fw fa-mail-reply',
                                   view_name='wf-cancel-publish.html',
                                   history_label=_("Publication canceled"),
                                   notify_roles={WEBMASTER_ROLE, PILOT_ROLE, MANAGER_ROLE, OWNER_ROLE},
                                   notify_message=_("cancelled the publication for content « {0} »"),
                                   order=1)

draft_to_published = Transition(transition_id='draft_to_published',
                                title=_("Publish"),
                                source=DRAFT,
                                destination=PUBLISHED,
                                permission=MANAGE_CONTENT_PERMISSION,
                                condition=can_manage_content,
                                action=publish_action,
                                prepared_transition=draft_to_prepublished,
                                menu_css_class='fa fa-fw fa-thumbs-o-up',
                                view_name='wf-publish.html',
                                history_label=_("Content published"),
                                notify_roles={WEBMASTER_ROLE, PILOT_ROLE, MANAGER_ROLE, OWNER_ROLE},
                                notify_message=_("published the content « {0} »"),
                                order=1)

published_to_archived = Transition(transition_id='published_to_archived',
                                   title=_("Archive content"),
                                   source=PUBLISHED,
                                   destination=ARCHIVED,
                                   permission=MANAGE_CONTENT_PERMISSION,
                                   condition=can_archive_content,
                                   action=archive_action,
                                   menu_css_class='fa fa-fw fa-archive',
                                   view_name='wf-archive.html',
                                   history_label=_("Content archived"),
                                   notify_roles={WEBMASTER_ROLE, PILOT_ROLE, MANAGER_ROLE, OWNER_ROLE},
                                   notify_message=_("archived content « {0} »"),
                                   order=2)

published_to_draft = Transition(transition_id='published_to_draft',
                                title=_("Create new version"),
                                source=PUBLISHED,
                                destination=DRAFT,
                                permission=CREATE_VERSION_PERMISSION,
                                condition=can_create_new_version,
                                action=clone_action,
                                menu_css_class='fa fa-fw fa-file-o',
                                view_name='wf-clone.html',
                                history_label=_("New version created"),
                                order=3)

wf_transitions = {init,
                  draft_to_prepublished,
                  prepublished_to_draft,
                  prepublished_to_published,
                  draft_to_published,
                  published_to_archived,
                  published_to_draft,
                  archived_to_draft,
                  delete}


@implementer(IAlertsWorkflow)
class AlertsWorkflow(BasicWorkflow):
    """PyAMS alerts workflow"""


wf = AlertsWorkflow(wf_transitions,
                    states=STATES_VOCABULARY,
                    initial_state=DRAFT,
                    update_states=UPDATE_STATES,
                    readonly_states=READONLY_STATES,
                    protected_states=PROTECTED_STATES,
                    manager_states=MANAGER_STATES,
                    published_states=PUBLISHED_STATES,
                    visible_states=VISIBLE_STATES,
                    waiting_states=WAITING_STATES,
                    retired_states=RETIRED_STATES,
                    archived_states=ARCHIVED_STATES,
                    auto_retired_state=ARCHIVED)


@utility_config(name='PyAMS alerts workflow', provides=IWorkflow)
class WorkflowUtility(object):
    """PyAMS alerts workflow utility

    This is an alerts workflow implementation for PyAMS contents.
    It only implements three states which are *draft*, *published* and *archived*.
    """

    def __new__(cls):
        return wf
