#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from itertools import islice, tee

from zope.schema.fieldproperty import FieldProperty

from pyams_content import _
from pyams_content.shared.view.interfaces import IViewsManager, IViewsMerger
from pyams_content.shared.view.portlet.interfaces import IViewItemsPortletSettings, SEARCH_EXCLUDED_ITEMS, \
    VIEW_DISPLAY_CONTEXT
from pyams_portal.interfaces import PREVIEW_MODE
from pyams_portal.portlet import Portlet, PortletSettings, portlet_config
from pyams_sequence.interfaces import ISequentialIdInfo
from pyams_sequence.reference import get_sequence_target
from pyams_utils.factory import factory_config
from pyams_utils.interfaces import VIEW_PERMISSION
from pyams_utils.interfaces.url import DISPLAY_CONTEXT
from pyams_utils.list import unique_iter
from pyams_utils.registry import get_utility
from pyams_utils.request import check_request
from pyams_workflow.interfaces import IWorkflow


VIEW_PORTLET_NAME = 'pyams_content.portlet.view'


@factory_config(provided=IViewItemsPortletSettings)
class ViewItemsPortletSettings(PortletSettings):
    """View items portlet settings"""

    title = FieldProperty(IViewItemsPortletSettings['title'])
    views = FieldProperty(IViewItemsPortletSettings['views'])
    views_context = FieldProperty(IViewItemsPortletSettings['views_context'])
    views_merge_mode = FieldProperty(IViewItemsPortletSettings['views_merge_mode'])
    limit = FieldProperty(IViewItemsPortletSettings['limit'])
    start = FieldProperty(IViewItemsPortletSettings['start'])
    force_canonical_url = FieldProperty(IViewItemsPortletSettings['force_canonical_url'])
    exclude_from_search = FieldProperty(IViewItemsPortletSettings['exclude_from_search'])
    first_page_only = FieldProperty(IViewItemsPortletSettings['first_page_only'])

    def get_views(self):
        views_manager = get_utility(IViewsManager)
        workflow = IWorkflow(views_manager)
        for oid in self.views or ():
            view = get_sequence_target(oid, state=workflow.visible_states)
            if view is not None:
                yield view

    def get_merger(self, request=None):
        if request is None:
            request = check_request()
        return request.registry.queryUtility(IViewsMerger, name=self.views_merge_mode)

    def get_items(self, request=None, limit=None, ignore_cache=False):
        if request is None:
            request = check_request()
        if self.views_context == VIEW_DISPLAY_CONTEXT:
            context = request.annotations.get(DISPLAY_CONTEXT, request.root)
        else:
            context = request.context
        if not ignore_cache:
            ignore_cache = request.annotations.get(PREVIEW_MODE, False)
        merger = self.get_merger(request)
        if merger is not None:
            start = int(request.params.get('vstart', 0))
            items = islice(unique_iter(merger.get_results(self.get_views(),
                                                          context,
                                                          ignore_cache=ignore_cache,
                                                          request=request)),
                           start + (self.start or 1) - 1,
                           limit or self.limit or 999)
            if (request is not None) and self.exclude_from_search:
                (excluded, items) = tee(items)
                excluded_items = request.annotations.get(SEARCH_EXCLUDED_ITEMS) or set()
                excluded_items |= set((ISequentialIdInfo(item).hex_oid for item in excluded))
                request.annotations[SEARCH_EXCLUDED_ITEMS] = excluded_items
            yield from items


@portlet_config(permission=VIEW_PERMISSION)
class ViewItemsPortlet(Portlet):
    """View items portlet"""

    name = VIEW_PORTLET_NAME
    label = _("View items")

    toolbar_image = None
    toolbar_css_class = 'fa fa-fw fa-2x fa-search'

    settings_factory = IViewItemsPortletSettings
