#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from collections import OrderedDict

from zope.interface import Attribute, Interface
from zope.schema import Bool, Choice, Int, Set
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_content.interfaces import VIEW_ORDER_VOCABULARY, VISIBLE_PUBLICATION_DATE_ORDER
from pyams_content.shared.common.interfaces import ISharedContent, ISharedTool, IWfSharedContent, \
    SHARED_CONTENT_TYPES_VOCABULARY
from pyams_content.shared.common.interfaces.types import ALL_DATA_TYPES_VOCABULARY
from pyams_sequence.interfaces import IInternalReferencesList
from pyams_sequence.schema import InternalReferencesListField
from pyams_thesaurus.schema import ThesaurusTermsListField


__docformat__ = 'restructuredtext'

from pyams_content import _


VIEW_CONTENT_TYPE = 'view'
VIEW_CONTENT_NAME = _('View')


class IViewsManager(ISharedTool):
    """Views manager interface"""


class IViewsManagerFactory(Interface):
    """Views manager factory interface"""


class IWfView(IWfSharedContent):
    """View interface"""

    select_context_path = Bool(title=_("Select context path?"),
                               description=_("If 'yes', only contents located inside context "
                                             "will be selected"),
                               required=True,
                               default=False)

    def get_content_path(self, context):
        """Get context path internal ID"""

    select_context_type = Bool(title=_("Select context type?"),
                               description=_("If 'yes', content type will be extracted from "
                                             "context"),
                               required=True,
                               default=False)

    selected_content_types = Set(title=_("Other content types"),
                                 description=_("Selected content types; leave empty for all"),
                                 value_type=Choice(vocabulary=SHARED_CONTENT_TYPES_VOCABULARY),
                                 required=False)

    def get_content_types(self, context):
        """Get content types for given context"""

    select_context_datatype = Bool(title=_("Select context data type?"),
                                   description=_("If 'yes', content data type (if available) "
                                                 "will be extracted from context"),
                                   required=True,
                                   default=False)

    selected_datatypes = Set(title=_("Other data types"),
                             description=_("Selected data types; leave empty for all"),
                             value_type=Choice(vocabulary=ALL_DATA_TYPES_VOCABULARY),
                             required=False)

    def get_data_types(self, context):
        """Get data types for given context"""

    excluded_content_types = Set(title=_("Excluded content types"),
                                 description=_("Excluded content types; leave empty for all"),
                                 value_type=Choice(vocabulary=SHARED_CONTENT_TYPES_VOCABULARY),
                                 required=False)

    def get_excluded_content_types(self, context):
        """Get excluded content types for given context"""

    excluded_datatypes = Set(title=_("Excluded data types"),
                             description=_("Excluded data types; leave empty for all"),
                             value_type=Choice(vocabulary=ALL_DATA_TYPES_VOCABULARY),
                             required=False)

    def get_excluded_data_types(self, context):
        """Get excluded data types for given context"""

    order_by = Choice(title=_("Order by"),
                      description=_("Property to use to sort results; publication date can be "
                                    "different from first publication date for contents which "
                                    "have been retired and re-published with a different "
                                    "publication date"),
                      vocabulary=VIEW_ORDER_VOCABULARY,
                      required=True,
                      default=VISIBLE_PUBLICATION_DATE_ORDER)

    reversed_order = Bool(title=_("Reversed order?"),
                          description=_("If 'yes', items order will be reversed"),
                          required=True,
                          default=True)

    limit = Int(title=_("Results count limit"),
                description=_("Maximum number of results that the view may retrieve"),
                required=False)

    age_limit = Int(title=_("Results age limit"),
                    description=_("If specified, contents whose publication date (given in "
                                  "days) is older than this limit will be ignored"),
                    required=False)

    is_using_context = Attribute("Check if view is using context settings")

    def get_results(self, context, sort_index=None, reverse=True, limit=None,
                    start=0, length=999, ignore_cache=False, get_count=False,
                    request=None, **kwargs):
        """Get results of catalog query"""


class IWfViewFactory(Interface):
    """View factory interface"""


class IView(ISharedContent):
    """Workflow managed view interface"""


class IViewSettings(Interface):
    """Base interface for view settings adapters"""

    is_using_context = Attribute("Check if view settings are using context")


END_PARAMS_MARKER = object()


class IViewQuery(Interface):
    """View query interface"""

    def get_params(self, context, request=None, **kwargs):
        """Get static view query params"""

    def get_results(self, context, sort_index, reverse, limit,
                    request=None, aggregates=None, settings=None, **kwargs):
        """Get tuple of limited results and total results count"""


class IViewUserQuery(Interface):
    """View user search query interface"""

    def get_user_params(self, request=None):
        """Get dynamic user query params"""


class IViewQueryExtension(Interface):
    """Base view query extension"""

    weight = Attribute("Extension weight")


class IViewQueryParamsExtension(IViewQueryExtension):
    """View query extension interface"""

    def get_params(self, context, request=None):
        """Add params to catalog query

        This method may return an iterator.
        If defined settings are such that no result can be found (for example, if the view is
        defined to use context themes but context doesn't have any), method can yield a "None"
        value which will cancel query execution.
        """


class IViewQueryFilterExtension(IViewQueryExtension):
    """View query filter extension"""

    def filter(self, context, items, request=None):
        """Filter items after catalog query"""


VIEW_REFERENCES_SETTINGS_KEY = 'pyams_content.view.references'

ALWAYS_REFERENCE_MODE = 'always'
IFEMPTY_REFERENCE_MODE = 'if_empty'
ONLY_REFERENCE_MODE = 'only'

REFERENCES_MODES = OrderedDict((
    (ALWAYS_REFERENCE_MODE, _("Always include selected internal references")),
    (IFEMPTY_REFERENCE_MODE, _("Include selected internal references only if view is empty")),
    (ONLY_REFERENCE_MODE, _("Include ONLY selected references (no search will be made)"))
))

REFERENCES_MODES_VOCABULARY = SimpleVocabulary([
    SimpleTerm(v, title=t)
    for v, t in REFERENCES_MODES.items()
])


class IViewInternalReferencesSettings(IViewSettings, IInternalReferencesList):
    """View internal references settings"""

    select_context_references = Bool(title=_("Select context references?"),
                                     description=_("If 'non', references imposed by the context "
                                                   "will not be used"),
                                     required=False,
                                     default=True)

    references = InternalReferencesListField(title=_("Other references"),
                                             description=_("List of internal references"),
                                             required=False)

    references_mode = Choice(title=_("Internal references usage"),
                             description=_("Specify how selected references are included into "
                                           "view results"),
                             vocabulary=REFERENCES_MODES_VOCABULARY,
                             required=True,
                             default=ALWAYS_REFERENCE_MODE)

    exclude_context = Bool(title=_("Exclude context?"),
                           description=_("If 'yes', context will be excluded from results list"),
                           required=True,
                           default=True)

    def get_references(self, context):
        """Get all references for given context"""


VIEW_TAGS_SETTINGS_KEY = 'pyams_content.view.tags'


class IViewTagsSettings(IViewSettings):
    """View tags settings"""

    select_context_tags = Bool(title=_("Select context tags?"),
                               description=_("If 'yes', tags will be extracted from context"),
                               required=False,
                               default=False)

    tags = ThesaurusTermsListField(title=_("Other tags"),
                                   required=False)

    def get_tags(self, context):
        """Get all tags for given context"""

    def get_tags_index(self, context):
        """Get all tags index values for given context"""


VIEW_THEMES_SETTINGS_KEY = 'pyams_content.view.themes'


class IViewThemesSettings(IViewSettings):
    """View themes settings"""

    select_context_themes = Bool(title=_("Select context themes?"),
                                 description=_("If 'yes', themes will be extracted from context"),
                                 required=False,
                                 default=False)

    themes = ThesaurusTermsListField(title=_("Other themes"),
                                     required=False)

    include_subthemes = Bool(title=_("Include all subthemes?"),
                             description=_("If 'yes', subthemes of selected themes will also "
                                           "be used to search contents"),
                             required=False,
                             default=False)

    def get_themes(self, context):
        """Get all themes for given context"""

    def get_themes_index(self, context):
        """Get all themes index values for given context"""


VIEW_COLLECTIONS_SETTINGS_KEY = 'pyams_content.view.collections'


class IViewCollectionsSettings(IViewSettings):
    """View collections settings"""

    select_context_collections = Bool(title=_("Select context collections?"),
                                      description=_("If 'yes', collections will be extracted "
                                                    "from context"),
                                      required=False,
                                      default=False)

    collections = ThesaurusTermsListField(title=_("Other collections"),
                                          required=False)

    def get_collections(self, context):
        """Get all collections for given context"""

    def get_collections_index(self, context):
        """Get all collections index values for given context"""


VIEWS_MERGERS_VOCABULARY = 'pyams_content.views.mergers'


class IViewsMerger(Interface):
    """Interface used to define views mergers

    Mergers are used to merge results of several views.
    """

    def get_results(self, views, context, ignore_cache=False, request=None):
        """Merge results of several views together"""
