#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.events import subscriber
from pyramid.path import DottedNameResolver
from z3c.form import field
from z3c.form.interfaces import IDataExtractedEvent
from zope.interface import Interface, Invalid
from zope.intid.interfaces import IIntIds
from zope.schema import Int, Text

from pyams_content.interfaces import IBaseContent, MANAGE_SITE_PERMISSION, MANAGE_TOOL_PERMISSION
from pyams_content.shared.common import IDashboardTypeColumnValue
from pyams_content.shared.common.interfaces import IBaseSharedTool
from pyams_content.shared.common.interfaces.zmi import IDashboardTable
from pyams_content.shared.common.zmi.manager import SharedToolPropertiesEditForm
from pyams_content.shared.site.interfaces import ISiteContainer, ISiteFolder, ISiteManager
from pyams_content.shared.site.zmi.widget import SiteManagerFoldersSelectorFieldWidget
from pyams_content.zmi.interfaces import ISiteTreeTable
from pyams_form.form import AJAXAddForm, AJAXEditForm, ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_form.interfaces.form import IInnerSubForm
from pyams_i18n.interfaces import II18n, INegotiator
from pyams_i18n.schema import I18nTextLineField
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.container import ITableElementEditor, ITableElementName
from pyams_skin.interfaces.viewlet import IMenuHeader, IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import DefaultElementEditorAdapter
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.registry import get_utility
from pyams_utils.traversing import get_parent
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, InnerAdminEditForm
from pyams_zmi.interfaces.menu import ISiteManagementMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='add-site-folder.menu', context=ISiteContainer, layer=IAdminLayer,
                view=ISiteTreeTable, manager=IToolbarAddingMenu,
                permission=MANAGE_SITE_PERMISSION, weight=10)
class SiteFolderAddMenu(ToolbarMenuItem):
    """Site folder add menu"""

    label = _("Add site folder...")
    label_css_class = 'fa fa-fw fa-folder-o'
    url = 'add-site-folder.html'
    modal_target = True


class ISiteFolderAddFormFields(Interface):
    """Site folder add form fields interface"""

    title = I18nTextLineField(title=_("Title"),
                              description=_("Visible label used to display content"),
                              required=True)

    parent = Int(title=_("Parent"),
                 description=_("Folder's parent"),
                 required=True)

    notepad = Text(title=_("Notepad"),
                   description=_("Internal information to be known about this content"),
                   required=False)


@pagelet_config(name='add-site-folder.html', context=ISiteContainer, layer=IPyAMSLayer,
                permission=MANAGE_SITE_PERMISSION)
@ajax_config(name='add-site-folder.json', context=ISiteContainer, layer=IPyAMSLayer,
             base=AJAXAddForm)
class SiteFolderAddForm(AdminDialogAddForm):
    """Site folder add form"""

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    legend = _("Add site folder")
    icon_css_class = 'fa fa-fw fa-folder-o'

    fields = field.Fields(ISiteFolderAddFormFields)
    fields['parent'].widgetFactory = SiteManagerFoldersSelectorFieldWidget

    edit_permission = MANAGE_SITE_PERMISSION

    def updateWidgets(self, prefix=None):
        super(SiteFolderAddForm, self).updateWidgets(prefix)
        if 'parent' in self.widgets:
            self.widgets['parent'].permission = MANAGE_SITE_PERMISSION

    def create(self, data):
        registry = self.request.registry
        factory = registry.settings.get('pyams_content.config.folder_factory')
        if factory:
            factory = DottedNameResolver().resolve(factory)
        else:
            manager = get_parent(self.context, ISiteManager)
            factory = manager.folder_factory
        return factory()

    def update_content(self, content, data):
        data = data.get(self, data)
        # initialize
        content.title = data['title']
        content.short_name = data['title']
        content.notepad = data['notepad']
        intids = get_utility(IIntIds)
        parent = intids.queryObject(data.get('parent'))
        if parent is not None:
            negotiator = get_utility(INegotiator)
            title = II18n(content).get_attribute('title', lang=negotiator.server_language)
            name = translate_string(title, force_lower=True, spaces='-')
            if name in parent:
                index = 1
                new_name = '{name}-{index:02}'.format(name=name, index=index)
                while new_name in parent:
                    index += 1
                    new_name = '{name}-{index:02}'.format(name=name, index=index)
                name = new_name
            parent[name] = content

    def add(self, content):
        pass

    def nextURL(self):
        return absolute_url(self.context, self.request, 'admin#site-tree.html')

    def get_ajax_output(self, changes):
        return {'status': 'reload'}


@subscriber(IDataExtractedEvent, form_selector=SiteFolderAddForm)
def handle_site_folder_add_form_data_extraction(event):
    """Handle site folder add form data extraction"""
    negotiator = get_utility(INegotiator)
    title = event.data.get('title', {}).get(negotiator.server_language)
    if not title:
        event.form.widgets.errors += (Invalid(_("You must provide a folder name for default "
                                                "server language!")),)


@adapter_config(context=(ISiteFolder, ISiteManagementMenu), provides=IMenuHeader)
class SiteFolderSiteManagementMenuHeader(ContextRequestAdapter):
    """Site folder site management menu header adapter"""

    header = _("Site folder management")


#
# Site folder edit form
#

@adapter_config(context=(ISiteFolder, IPyAMSLayer, IDashboardTable),
                provides=IDashboardTypeColumnValue)
def site_folder_dashboard_type(context, request, view):
    """Site folder dashboard type column adapter"""
    return request.localizer.translate(context.content_name)


@adapter_config(context=(ISiteFolder, IPyAMSLayer), provides=ITableElementName)
class SiteFolderTableElementNameAdapter(ContextRequestAdapter):
    """Site folder table element name adapter"""

    @property
    def name(self):
        return II18n(self.context).query_attribute('short_name', request=self.request)


@adapter_config(context=(ISiteFolder, IAdminLayer, Interface), provides=ITableElementEditor)
class SiteFolderTableElementEditorAdapter(DefaultElementEditorAdapter):
    """Site folder table element editor adapter"""

    view_name = 'admin#site-tree.html'
    modal_target = False


@pagelet_config(name='properties.html', context=ISiteFolder, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='properties.json', context=ISiteFolder, layer=IPyAMSLayer)
class SiteFolderPropertiesEditForm(SharedToolPropertiesEditForm):
    """Site folder properties edit form"""

    legend = _("Site folder properties")

    fields = field.Fields(ISiteFolder).select('title', 'short_name', 'header', 'description',
                                              'notepad') + \
        field.Fields(IBaseSharedTool).select('shared_content_workflow')

    def get_ajax_output(self, changes):
        updated = changes.get(IBaseContent, ())
        if 'title' in updated:
            return {
                'status': 'redirect',
                'message': self.request.localizer.translate(self.successMessage)
            }
        else:
            return AJAXEditForm.get_ajax_output(self, changes)


@adapter_config(name='navigation',
                context=(ISiteFolder, IPyAMSLayer, SiteFolderPropertiesEditForm),
                provides=IInnerSubForm)
class SiteFolderNavigationPropertiesEditForm(InnerAdminEditForm):
    """Site folder navigation properties edit form"""

    prefix = 'navigation_form.'

    css_class = 'form-group'
    padding_class = ''
    fieldset_class = 'bordered margin-top-10 padding-y-5'

    legend = None
    main_group_legend = _("Navigation properties")
    main_group_class = 'inner switcher no-y-padding'

    fields = field.Fields(ISiteFolder).select('visible_in_list', 'navigation_title',
                                              'navigation_mode')

    weight = 5

    def check_mode(self):
        if self.parent_form is not None:
            self.mode = self.parent_form.mode

    def updateGroups(self):
        self.add_group(NamedWidgetsGroup(self, 'navigation', self.widgets,
                                         ('visible_in_list', 'navigation_title',
                                          'navigation_mode'),
                                         fieldset_class=self.fieldset_class,
                                         legend=self.main_group_legend,
                                         css_class=self.main_group_class,
                                         switch=True,
                                         display_mode='always'))
        super(SiteFolderNavigationPropertiesEditForm, self).updateGroups()
