#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from uuid import uuid4

from pyramid.decorator import reify
from pyramid.path import DottedNameResolver
from z3c.form import field
from zope.intid.interfaces import IIntIds
from zope.lifecycleevent import ObjectCreatedEvent
from zope.schema import Int

from pyams_content.interfaces import CREATE_CONTENT_PERMISSION
from pyams_content.shared.common.zmi import SharedContentAJAXAddForm
from pyams_content.shared.common.zmi.types import TypedSharedContentAddForm
from pyams_content.shared.site import IWfSiteTopic
from pyams_content.shared.site.interfaces import ISiteContainer, ISiteManager
from pyams_content.shared.site.zmi.widget import SiteManagerFoldersSelectorFieldWidget
from pyams_content.zmi.interfaces import ISiteTreeTable
from pyams_form.form import ajax_config
from pyams_i18n.interfaces import II18nManager, INegotiator
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.registry import get_utility
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url, generate_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_workflow.interfaces import IWorkflowInfo, IWorkflowVersions
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='add-topic.menu', context=ISiteContainer, layer=IAdminLayer,
                view=ISiteTreeTable, manager=IToolbarAddingMenu,
                permission=CREATE_CONTENT_PERMISSION, weight=20)
class TopicAddMenu(ToolbarMenuItem):
    """Topic add menu"""

    label = _("Add topic...")
    label_css_class = 'fa fa-fw fa-file-o'
    url = 'add-topic.html'
    modal_target = True


class ITopicAddFormFields(IWfSiteTopic):
    """Topic add form fields interface"""

    parent = Int(title=_("Parent"),
                 description=_("Topic's parent"),
                 required=True)


@pagelet_config(name='add-topic.html', context=ISiteContainer, layer=IPyAMSLayer,
                permission=CREATE_CONTENT_PERMISSION)
@pagelet_config(name='add-shared-content.html', context=ISiteContainer, layer=IPyAMSLayer,
                permission=CREATE_CONTENT_PERMISSION)
@ajax_config(name='add-topic.json', context=ISiteContainer, layer=IPyAMSLayer,
             base=SharedContentAJAXAddForm)
class TopicAddForm(TypedSharedContentAddForm):
    """Topic add form"""

    legend = _("Add topic")

    fields = field.Fields(ITopicAddFormFields).select('title', 'data_type', 'parent', 'notepad')
    fields['parent'].widgetFactory = SiteManagerFoldersSelectorFieldWidget

    edit_permission = CREATE_CONTENT_PERMISSION

    __target = None

    def updateWidgets(self, prefix=None):
        super(TopicAddForm, self).updateWidgets(prefix)
        if 'parent' in self.widgets:
            self.widgets['parent'].permission = CREATE_CONTENT_PERMISSION

    @reify
    def content_factory(self):
        registry = self.request.registry
        factory = registry.settings.get('pyams_content.config.topic_factory')
        if factory:
            factory = DottedNameResolver().resolve(factory)
        else:
            manager = get_parent(self.context, ISiteManager)
            factory = manager.topic_content_factory
        return factory

    def create(self, data):
        return self.content_factory.content_class()

    def update_content(self, content, data):
        data = data.get(self, data)
        # initialize content fields
        lang = get_utility(INegotiator).server_language
        content.creator = self.request.principal.id
        content.owner = self.request.principal.id
        content.title = data['title']
        content.short_name = content.title.copy()
        content.content_url = generate_url(content.title.get(lang, ''))
        content.notepad = data.get('notepad')
        # get parent
        intids = get_utility(IIntIds)
        parent = intids.queryObject(data.get('parent'))
        if parent is not None:
            languages = II18nManager(parent).languages
            if languages:
                II18nManager(content).languages = languages.copy()
            wf_parent = self.content_factory()
            self.request.registry.notify(ObjectCreatedEvent(wf_parent))
            uuid = str(uuid4())
            parent[uuid] = wf_parent
            IWorkflowVersions(wf_parent).add_version(content, None)
            IWorkflowInfo(content).fire_transition('init', comment=content.notepad)
            self.__target = content

    def add(self, content):
        pass

    def nextURL(self):
        return absolute_url(self.__target, self.request, 'admin')
