#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
import locale

from pyramid.events import subscriber
from zope.component.interfaces import ISite
from zope.container.ordered import OrderedContainer
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectRemovedEvent
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_content.component.illustration import IIllustrationTarget, ILinkIllustrationTarget
from pyams_content.component.paragraph.interfaces import IParagraphFactorySettings
from pyams_content.component.theme.interfaces import ICollectionsManagerTarget, \
    IThemesManagerTarget
from pyams_content.features.footer.interfaces import IFooterTarget
from pyams_content.features.header.interfaces import IHeaderTarget
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.interfaces import MANAGE_SITE_PERMISSION
from pyams_content.reference.pictograms.interfaces import IPictogramManagerTarget
from pyams_content.root.interfaces import ISiteRoot
from pyams_content.shared.common.interfaces import ISharedContentFactory
from pyams_content.shared.common.manager import BaseSharedTool
from pyams_content.shared.common.types import TypedSharedToolMixin
from pyams_content.shared.site import SiteTopic
from pyams_content.shared.site.container import SiteContainerMixin
from pyams_content.shared.site.folder import SiteFolder
from pyams_content.shared.site.interfaces import ISiteContainer, ISiteFolderFactory, \
    ISiteManager, ISiteManagerFactory
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext
from pyams_skin.skin import UserSkinnableContent
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.registry import get_utilities_for
from pyams_utils.request import query_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config


__docformat__ = 'restructuredtext'

from pyams_content import _


@implementer(ISiteManager, IParagraphFactorySettings, IThemesManagerTarget,
             IPictogramManagerTarget, IIllustrationTarget, ILinkIllustrationTarget,
             IPortalContext, IHeaderTarget, IFooterTarget, IPreviewTarget)
class SiteManager(SiteContainerMixin, OrderedContainer, BaseSharedTool, TypedSharedToolMixin,
                  UserSkinnableContent):
    """Site manager persistent class"""

    header = FieldProperty(ISiteManager['header'])
    description = FieldProperty(ISiteManager['description'])
    notepad = FieldProperty(ISiteManager['notepad'])
    navigation_mode = FieldProperty(ISiteManager['navigation_mode'])
    indexation_mode = FieldProperty(ISiteManager['indexation_mode'])

    allowed_paragraphs = FieldProperty(IParagraphFactorySettings['allowed_paragraphs'])
    auto_created_paragraphs = FieldProperty(IParagraphFactorySettings['auto_created_paragraphs'])

    sequence_name = ''  # use default sequence generator
    sequence_prefix = ''

    content_name = _("Site manager")

    @property
    def folder_factory(self):
        return ISiteFolderFactory(self, SiteFolder)

    @property
    def topic_content_factory(self):
        return ISharedContentFactory(self, None)

    @property
    def topic_content_type(self):
        factory = self.topic_content_factory
        if factory is not None:
            return factory.content_class.content_type

    def is_deletable(self):
        for element in self.values():
            if not element.is_deletable():
                return False
        return True


@adapter_config(context=ISiteRoot, provides=ISiteManagerFactory)
def site_root_site_manager_factory(context):
    return SiteManager


@adapter_config(context=ISiteManager, provides=ISiteFolderFactory)
def site_manager_folder_factory(context):
    return SiteFolder


@adapter_config(context=ISiteContainer, provides=ISharedContentFactory)
def site_manager_shared_content_factory(context):
    return SiteTopic


@subscriber(IObjectAddedEvent, context_selector=ISiteManager)
def handle_added_site_manager(event):
    """Register site manager when added"""
    site = get_parent(event.newParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, ISiteManager, name=event.object.__name__)


@subscriber(IObjectRemovedEvent, context_selector=ISiteManager)
def handle_deleted_site_manager(event):
    """Un-register site manager when deleted"""
    site = get_parent(event.oldParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.unregisterUtility(event.object, ISiteManager, name=event.object.__name__)


@adapter_config(context=ISiteManager, provides=IFormContextPermissionChecker)
class SiteManagerPermissionChecker(ContextAdapter):
    """Site manager edit permission checker"""

    edit_permission = MANAGE_SITE_PERMISSION


@vocabulary_config(name='PyAMS site managers')
class SiteManagerVocabulary(SimpleVocabulary):
    """Site manager vocabulary"""

    interface = ISiteManager

    def __init__(self, context):
        request = query_request()
        terms = [
            SimpleTerm(v, title=II18n(t).query_attribute('title', request=request))
            for v, t in get_utilities_for(self.interface)
        ]
        super(SiteManagerVocabulary, self).__init__(sorted(terms,
                                                           key=lambda x: locale.strxfrm(x.title)))
