#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from zope.container.contained import Contained
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.illustration import ILinkIllustrationTarget
from pyams_content.component.links import ExternalLink, InternalLink
from pyams_content.features.menu import IDynamicMenu
from pyams_content.shared.site.interfaces import IContentLink, IExternalContentLink, ISiteElementNavigation, ISiteLink
from pyams_sequence.reference import get_reference_target
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.zodb import volatile_property
from pyams_workflow.interfaces import IWorkflow, IWorkflowPublicationInfo, IWorkflowState, IWorkflowVersion, \
    IWorkflowVersions

from pyams_content import _


@implementer(ISiteLink)
class SiteLink(Persistent, Contained):
    """Site link persistent class"""

    navigation_title = FieldProperty(IContentLink['navigation_title'])
    show_header = FieldProperty(IContentLink['show_header'])
    navigation_header = FieldProperty(IContentLink['navigation_header'])
    visible = FieldProperty(IContentLink['visible'])

    @staticmethod
    def is_deletable():
        return True


#
# Internal content link
#

@implementer(IContentLink, ILinkIllustrationTarget)
class ContentLink(SiteLink):
    """Content link persistent class

    A 'content link' is a link to another content, which may be stored anywhere (same site,
    another site or in any shared tool).
    """

    reference = FieldProperty(IContentLink['reference'])

    content_name = _("Content link")

    @volatile_property
    def target(self):
        target = get_reference_target(self.reference)
        if IWorkflowVersion.providedBy(target):
            workflow = IWorkflow(target, None)
            if workflow is not None:
                versions = IWorkflowVersions(target).get_versions(workflow.visible_states, sort=True)
                if not versions:
                    versions = IWorkflowVersions(target).get_last_versions()
                if versions:
                    target = versions[-1]
        return target

    def get_target(self, state=None):
        target = self.target
        if (target is None) and self.reference:
            # verify that volatile property doesn't have to be reset
            del self.target
            target = self.target
        return target


@adapter_config(context=IContentLink, provides=IDynamicMenu)
def content_link_dynamic_menu_factory(context):
    target = context.get_target()
    if target is not None:
        result = InternalLink()
        result.title = context.navigation_title.copy() if context.navigation_title else {}
        result.reference = context.reference
        return result


@adapter_config(context=(IContentLink, IPyAMSLayer), provides=ISiteElementNavigation)
class ContentLinkNavigationAdapter(ContextRequestAdapter):
    """Content link navigation adapter"""

    @property
    def visible(self):
        if not self.context.visible:
            return False
        target = self.context.target
        return (target is not None) and IWorkflowPublicationInfo(target).is_visible(self.request)


@adapter_config(context=IContentLink, provides=IWorkflow)
def content_link_workflow_info(context):
    """Content link workflow info"""
    target = context.get_target()
    if target is not None:
        return IWorkflow(target, None)


@adapter_config(context=IContentLink, provides=IWorkflowState)
def content_link_state_info(context):
    """Content link workflow state info"""
    target = context.get_target()
    if target is not None:
        return IWorkflowState(target, None)


@adapter_config(context=IContentLink, provides=IWorkflowPublicationInfo)
def content_link_publication_info(context):
    """Content link publication info"""
    target = context.get_target()
    if target is not None:
        return IWorkflowPublicationInfo(target, None)


#
# External content link
#

@implementer(IExternalContentLink, ILinkIllustrationTarget)
class ExternalContentLink(SiteLink):
    """External link persistent class"""

    url = FieldProperty(IExternalContentLink['url'])

    content_name = _("External content link")


@adapter_config(context=IExternalContentLink, provides=IDynamicMenu)
def external_content_link_dynamic_menu_factory(context):
    """External content link dynamic menu factory"""
    result = ExternalLink()
    result.title = context.navigation_title.copy() if context.navigation_title else {}
    result.url = context.url
    return result


@adapter_config(context=(IExternalContentLink, IPyAMSLayer), provides=ISiteElementNavigation)
class ExternalContentLinkNavigationAdapter(ContextRequestAdapter):
    """External content link navigation adapter"""

    @property
    def visible(self):
        return self.context.visible
