#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import implementer

from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
    BaseParagraphAddForm, BaseParagraphAddMenu, BaseParagraphPropertiesEditForm, IParagraphContainerView, \
    IParagraphInnerEditFormButtons
from pyams_content.component.paragraph.zmi.interfaces import IParagraphInnerEditor
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common import IWfSharedContent
from pyams_content.shared.logo.interfaces import ILogosParagraph, LOGOS_PARAGRAPH_TYPE
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_widget_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import viewlet_config

from pyams_content import _


@viewlet_config(name='add-logos-paragraph.menu', context=IParagraphContainerTarget, view=IParagraphContainerView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=600)
class LogosParagraphAddMenu(BaseParagraphAddMenu):
    """Logos paragraph add menu"""

    label = _("Logos...")
    label_css_class = 'fa fa-fw fa-th-large'
    url = 'add-logos-paragraph.html'
    paragraph_type = LOGOS_PARAGRAPH_TYPE


@pagelet_config(name='add-logos-paragraph.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-logos-paragraph.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class LogosParagraphAddForm(BaseParagraphAddForm):
    """Logos paragraph add form"""

    legend = _("Add new logos paragraph")

    content_interface = ILogosParagraph


@pagelet_config(name='properties.html', context=ILogosParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=ILogosParagraph, layer=IPyAMSLayer)
class LogosParagraphPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Logos paragraph properties edit form"""

    prefix = 'logos_properties.'

    @property
    def title(self):
        content = get_parent(self.context, IWfSharedContent)
        return II18n(content).query_attribute('title', request=self.request)

    legend = _("Edit paragraph properties")

    content_interface = ILogosParagraph

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if 'renderer' in changes.get(ILogosParagraph, ()):
            output.setdefault('events', []).append(
                get_json_widget_refresh_event(self.context, self.request,
                                              LogosParagraphInnerEditForm, 'renderer'))
        return output


@adapter_config(context=(ILogosParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=ILogosParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm)
class LogosParagraphInnerEditForm(LogosParagraphPropertiesEditForm):
    """Logos paragraph properties inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()
