#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify
from pyramid.view import view_config
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn, IValues
from zope.interface import alsoProvides, implementer

from pyams_content.component.association.interfaces import IAssociationContainer, IAssociationInfo
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.zmi import WfModifiedContentColumnMixin
from pyams_content.shared.imagemap.interfaces import IWfImageMap
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces import IInnerPage
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import BaseTable, I18nColumn, TrashColumn
from pyams_skin.viewlet.menu import MenuItem
from pyams_template.template import template_config
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView

__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='areas.submenu', context=IWfImageMap, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=MANAGE_CONTENT_PERMISSION, weight=210)
class ImagemapAreasMenu(MenuItem):
    """Image map areas menu"""

    label = _("Image areas")
    icon_class = 'fa-location-arrow'
    url = '#areas.html'


@pagelet_config(name='areas.html', context=IWfImageMap, layer=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION)
@template_config(template='templates/container.pt', layer=IPyAMSLayer)
@implementer(IInnerPage)
class ImagemapAreasContainerView(AdminView):
    """Image map areas container view"""

    title = _("Image map areas")

    def __init__(self, context, request):
        super(ImagemapAreasContainerView, self).__init__(context, request)
        self.areas_table = ImagemapAreasTable(context, request)

    def update(self):
        super(ImagemapAreasContainerView, self).update()
        self.areas_table.update()


class ImagemapAreasTable(BaseTable):
    """Image map areas table"""

    prefix = 'imagemaps'

    hide_header = True
    cssClasses = {'table': 'table table-bordered table-striped table-hover table-tight'}

    def __init__(self, context, request):
        super(ImagemapAreasTable, self).__init__(context, request)
        self.object_data = {'ams-widget-toggle-button': 'false'}
        alsoProvides(self, IObjectData)

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-location': absolute_url(self.context, self.request),
            'data-ams-datatable-sort': 'false',
            'data-ams-datatable-pagination': 'false'
        })
        return attributes

    @reify
    def values(self):
        return list(super(ImagemapAreasTable, self).values)

    def render(self):
        translate = self.request.localizer.translate
        has_value = False
        image = self.context.image
        if image:
            for value in image.values():
                if value:
                    has_value = True
                    break
        if not has_value:
            return translate(_("No currently defined image."))
        if not self.values:
            return translate(_("No currently defined area."))
        return super(ImagemapAreasTable, self).render()


@adapter_config(name='name', context=(IWfImageMap, IPyAMSLayer, ImagemapAreasTable), provides=IColumn)
class ImagemapAreasContainerNameColumn(I18nColumn, WfModifiedContentColumnMixin, GetAttrColumn):
    """Image map areas container name column"""

    _header = _("Title")

    weight = 10

    def getValue(self, obj):
        title = II18n(obj).query_attribute('title', request=self.request)
        if not title:
            item = IAssociationContainer(self.context).get(obj.link)
            if item is not None:
                title = IAssociationInfo(item).user_title
        return title


@adapter_config(name='target', context=(IWfImageMap, IPyAMSLayer, ImagemapAreasTable), provides=IColumn)
class ImagemapAreasContainerTargetColumn(I18nColumn, GetAttrColumn):
    """Image map areas container target URL column"""

    _header = _("Link target")

    weight = 20

    def getValue(self, obj):
        item = IAssociationContainer(self.context).get(obj.link)
        if item is not None:
            return IAssociationInfo(item).inner_title


@adapter_config(name='trash', context=(IWfImageMap, IPyAMSLayer, ImagemapAreasTable), provides=IColumn)
class ImagemapAreasContainerTrashColumn(ProtectedFormObjectMixin, TrashColumn):
    """Image map areas container trash column"""


@adapter_config(context=(IWfImageMap, IPyAMSLayer, ImagemapAreasTable), provides=IValues)
class ImagemapAreasValues(ContextRequestViewAdapter):
    """Image map areas caontainer values"""

    @property
    def values(self):
        return self.context.areas.values()


@view_config(name='delete-element.json', context=IWfImageMap, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, renderer='json', xhr=True)
def delete_imagemap_area(request):
    """Delete image map area"""
    translate = request.localizer.translate
    name = request.params.get('object_name')
    if not name:
        return {
            'status': 'message',
            'messagebox': {
                'status': 'error',
                'content': translate(_("No provided object_name argument!"))
            }
        }
    if '++' in name:
        try:
            name = name.split('++', 2)[2]
        except IndexError:
            return {
                'status': 'message',
                'messagebox': {
                    'status': 'error',
                    'content': translate(_("Bad query object_name parameter value!"))
                }
            }
    if name not in request.context.areas:
        return {
            'status': 'message',
            'messagebox': {
                'status': 'error',
                'content': translate(_("Given area name doesn't exist!"))
            }
        }
    request.context.remove_area(name)
    return {'status': 'success'}
