#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
from pyams_skin.event import get_json_table_row_refresh_event
from pyams_utils.traversing import get_parent

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.imagemap.interfaces import IWfImageMap, IImageMapArea
from pyams_i18n.interfaces import II18n
from pyams_skin.interfaces.viewlet import IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer

# import packages
from pyams_content.shared.imagemap import ImageMapArea
from pyams_content.shared.imagemap.zmi.container import ImagemapAreasContainerView, ImagemapAreasTable
from pyams_content.shared.imagemap.zmi.widget import ImgareaInputFieldWidget
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm
from z3c.form import field

from pyams_content import _


@viewlet_config(name='add-imagemap-area.action', context=IWfImageMap, layer=IPyAMSLayer,
                view=ImagemapAreasContainerView, manager=IWidgetTitleViewletManager,
                permission=MANAGE_CONTENT_PERMISSION, weight=1)
class ImagemapAreaAddAction(ToolbarAction):
    """Image map area add menu"""

    label = _("Add image area")

    url = 'add-area.html'
    modal_target = True

    def __new__(cls, context, request, view, manager):
        image = context.image
        if image:
            for value in image.values():
                if value:
                    return ToolbarAction.__new__(cls)
        return None


@pagelet_config(name='add-area.html', context=IWfImageMap, layer=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-area.json', context=IWfImageMap, layer=IPyAMSLayer, base=AJAXAddForm)
class ImagemapAreaAddForm(AdminDialogAddForm):
    """Imagemap area add form"""

    legend = _("Add new image area")

    label_css_class = 'control-label col-md-2'
    input_css_class = 'col-md-10'

    @property
    def dialog_class(self):
        image = II18n(self.context).query_attribute('image', request=self.request)
        if image is not None:
            width = image.get_image_size()[0]
            if width > 1000:
                return 'modal-max'
            elif width > 800:
                return 'modal-xlarge'
            else:
                return 'modal-large'
        else:
            return 'modal-large'

    fields = field.Fields(IImageMapArea)
    fields['area'].widgetFactory = ImgareaInputFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def create(self, data):
        return ImageMapArea()

    def add(self, object):
        self.context.add_area(object)

    def nextURL(self):
        return 'areas.html'


@pagelet_config(name='properties.html', context=IImageMapArea, layer=IPyAMSLayer, permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IImageMapArea, layer=IPyAMSLayer)
class ImagemapAreaPropertiesEditForm(AdminDialogEditForm):
    """Image map area properties edit form"""

    prefix = 'area_properties.'

    legend = _("Edit image map properties")

    label_css_class = 'control-label col-md-2'
    input_css_class = 'col-md-10'

    @property
    def dialog_class(self):
        image = II18n(self.context.__parent__).query_attribute('image', request=self.request)
        if image is not None:
            width = image.get_image_size()[0]
            if width > 1000:
                return 'modal-max'
            elif width > 800:
                return 'modal-xlarge'
            else:
                return 'modal-large'
        else:
            return 'modal-large'

    fields = field.Fields(IImageMapArea)
    fields['area'].widgetFactory = ImgareaInputFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if changes:
            container = get_parent(self.context, IWfImageMap)
            output.setdefault('events', []).append(
                get_json_table_row_refresh_event(container, self.request, ImagemapAreasTable, self.context))
        return output
