#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.links import InternalReferenceMixin
from pyams_content.component.paragraph import BaseParagraph, BaseParagraphContentChecker, BaseParagraphFactory
from pyams_content.component.paragraph.interfaces import IParagraphFactory
from pyams_content.features.checker.interfaces import ERROR_VALUE, IContentChecker, MISSING_LANG_VALUE, MISSING_VALUE
from pyams_content.features.json import IJSONConverter
from pyams_content.features.renderer import RenderersVocabulary
from pyams_content.shared.imagemap.interfaces import IImageMapParagraph, IMAGEMAP_PARAGRAPH_NAME, \
    IMAGEMAP_PARAGRAPH_RENDERERS, IMAGEMAP_PARAGRAPH_TYPE
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import NullAdapter, adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, utility_config
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config
from pyams_workflow.interfaces import IWorkflow, IWorkflowState

from pyams_content import _
from pyams_workflow.workflow import NullCondition


@factory_config(provided=IImageMapParagraph)
class ImageMapParagraph(BaseParagraph, InternalReferenceMixin):
    """Image map paragraph"""

    factory_name = IMAGEMAP_PARAGRAPH_TYPE
    icon_class = 'fa-location-arrow'
    icon_hint = IMAGEMAP_PARAGRAPH_NAME

    reference = FieldProperty(IImageMapParagraph['reference'])
    renderer = FieldProperty(IImageMapParagraph['renderer'])


@utility_config(name=IMAGEMAP_PARAGRAPH_TYPE, provides=IParagraphFactory)
class ImageMapParagraphFactory(BaseParagraphFactory):
    """Image map paragraph factory"""

    name = IMAGEMAP_PARAGRAPH_NAME
    content_type = ImageMapParagraph
    secondary_menu = True


@adapter_config(context=IImageMapParagraph, provides=IContentChecker)
class ImageMapParagraphContentChecker(BaseParagraphContentChecker):
    """ImageMap paragraph content checker"""

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language, )
        i18n = II18n(self.context)
        for lang in langs:
            value = i18n.get_attribute('title', lang, request)
            if not value:
                field_title = translate(IImageMapParagraph['title'].title)
                if len(langs) == 1:
                    output.append(translate(MISSING_VALUE).format(field=field_title))
                else:
                    output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        field_title = translate(IImageMapParagraph['reference'].title)
        if not self.context.reference:
            output.append(translate(ERROR_VALUE).format(
                field=field_title,
                message=translate(_("no selected image map"))))
        else:
            target = self.context.get_target()
            if target is None:
                output.append(translate(ERROR_VALUE).format(
                    field=field_title,
                    message=translate(_("image map '{0}' can't be found")).reference(self.context.reference)))
            else:
                workflow = IWorkflow(target, None)
                if workflow is not None:
                    workflow_state = IWorkflowState(target)
                    if workflow_state.state not in workflow.published_states:
                        output.append(translate(ERROR_VALUE).format(
                            field=field_title,
                            message=translate(_("image map '{0}' is not published")).format(
                                II18n(target).query_attribute('title', request=request))))
        return output


@vocabulary_config(name=IMAGEMAP_PARAGRAPH_RENDERERS)
class ImageMapParagraphRendererVocabulary(RenderersVocabulary):
    """Image map paragraph renderers vocabulary"""

    content_interface = IImageMapParagraph


@adapter_config(required=(IImageMapParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONImageMapParagraphConverter(NullAdapter):
    """JSON image map paragraph converter"""
