#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.path import DottedNameResolver
from z3c.form import field

from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION, MANAGE_TOOL_PERMISSION
from pyams_content.root.interfaces import ISiteRoot
from pyams_content.shared.common import IBaseSharedTool, IDashboardTypeColumnValue
from pyams_content.shared.common.interfaces.zmi import IDashboardTable
from pyams_content.shared.common.zmi.manager import SharedToolPropertiesEditForm
from pyams_content.shared.hub.interfaces import IHubManager, IHubManagerFactory
from pyams_content.shared.hub.manager import HubManager
from pyams_content.zmi.interfaces import ISiteTreeTable
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.viewlet import IMenuHeader, IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm
from pyams_zmi.interfaces.menu import ISiteManagementMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from pyams_content import _


@adapter_config(context=(IHubManager, IPyAMSLayer, IDashboardTable),
                provides=IDashboardTypeColumnValue)
def hub_manager_dashboard_type(context, request, view):
    """Site manager dashboard type column adapter"""
    return request.localizer.translate(context.content_name)


@adapter_config(context=(IHubManager, ISiteManagementMenu), provides=IMenuHeader)
class HubManagerSiteManagementMenuHeader(ContextRequestAdapter):
    """Hub manager site management menu header adapter"""

    header = _("Hub management")


@viewlet_config(name='add-hub-manager.menu', context=ISiteRoot, layer=IAdminLayer,
                view=ISiteTreeTable, manager=IToolbarAddingMenu,
                permission=MANAGE_SITE_ROOT_PERMISSION)
class HubManagerAddMenu(ToolbarMenuItem):
    """Hub manager add menu"""

    label = _("Add hub manager")
    label_css_class = 'fa fa-fw fa-shopping-basket'
    url = 'add-hub-manager.html'
    modal_target = True


@pagelet_config(name='add-hub-manager.html', context=ISiteRoot, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='add-hub-manager.json', context=ISiteRoot, layer=IPyAMSLayer,
             permission=MANAGE_SITE_ROOT_PERMISSION, base=AJAXAddForm)
class HubManagerAddForm(AdminDialogAddForm):
    """Hub manager add form"""

    title = _("Hub manager")
    legend = _("Add hub manager")
    icon_css_class = 'fa fa-fw fa-shopping-basket'

    fields = field.Fields(IHubManager).select('title', 'short_name')
    edit_permission = None

    def create(self, data):
        factory = self.request.registry.settings.get('pyams_content.config.hub_factory')
        if factory:
            factory = DottedNameResolver().resolve(factory)
        else:
            factory = IHubManagerFactory(self.context, HubManager)
        return factory()

    def add(self, object):
        short_name = II18n(object).query_attribute('short_name', request=self.request)
        name = translate_string(short_name, force_lower=True, spaces='-')
        self.context[name] = object

    def nextURL(self):
        return absolute_url(self.context, self.request, 'site-tree.html')


@pagelet_config(name='properties.html', context=IHubManager, layer=IPyAMSLayer,
                permission=MANAGE_TOOL_PERMISSION)
@ajax_config(name='properties.json', context=IHubManager, layer=IPyAMSLayer)
class HubManagerPropertiesEditForm(SharedToolPropertiesEditForm):
    """Hub manager properties edit form"""

    legend = _("Hub manager properties")

    fields = field.Fields(IHubManager).select('title', 'short_name', 'header',
                                              'navigation_header', 'description', 'notepad') + \
        field.Fields(IBaseSharedTool).select('shared_content_workflow') + \
        field.Fields(IHubManager).select('navigation_mode', 'indexation_mode')
