#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
from pyramid.events import subscriber
from z3c.form import field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from z3c.form.interfaces import IDataExtractedEvent, INPUT_MODE
from zope.interface import Interface, Invalid

from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.zmi.properties import SharedContentPropertiesEditForm
from pyams_content.shared.form.interfaces import IWfForm
from pyams_form.form import ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_form.interfaces.form import IInnerSubForm
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_zmi.form import InnerAdminEditForm


__docformat__ = 'restructuredtext'

from pyams_content import _


@pagelet_config(name='properties.html', context=IWfForm, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json', context=IWfForm, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION)
class FormPropertiesEditForm(SharedContentPropertiesEditForm):
    """Form properties edit form"""

    interface = IWfForm
    fieldnames = ('title', 'short_name', 'content_url', 'alt_title',
                  'header', 'description', 'notepad')


@adapter_config(name='form-settings',
                context=(IWfForm, IPyAMSLayer, FormPropertiesEditForm),
                provides=IInnerSubForm)
class FormPropertiesInnerEditForm(InnerAdminEditForm):
    """Form properties edit form extension"""

    prefix = 'form_properties.'

    legend = _("Main form settings")
    fieldset_class = 'bordered no-x-margin margin-y-10'

    fields = field.Fields(IWfForm).select('form_header', 'user_title', 'auth_only',
                                          'submit_label', 'submit_message', 'handler',
                                          'override_captcha', 'client_captcha_key',
                                          'server_captcha_key',
                                          'rgpd_consent', 'rgpd_warning', 'rgpd_user_rights')
    fields['override_captcha'].widgetFactory = SingleCheckBoxFieldWidget
    fields['rgpd_consent'].widgetFactory = SingleCheckBoxFieldWidget

    weight = 1

    def updateWidgets(self, prefix=None):
        super(FormPropertiesInnerEditForm, self).updateWidgets(prefix)
        if self.mode == INPUT_MODE:
            translate = self.request.localizer.translate
            if 'submit_message' in self.widgets:
                self.widgets['submit_message'].after_widget_notice = \
                    '<div class="alert-info padding-5">{0}</div>'.format(
                        translate(_("Text samples:<br />"
                                    "- Thank you for taking contact with us, your message has been "
                                    "transmitted.")))
            if 'rgpd_warning' in self.widgets:
                self.widgets['rgpd_warning'].after_widget_notice = \
                    '<div class="alert-info padding-5">{0}</div>'.format(
                        translate(_("Text samples:<br />"
                                    "- By submitting this form, I agree that the information "
                                    "entered may be used for the purpose of my request and the "
                                    "business relationship that may result from it.")))
            if 'rgpd_user_rights' in self.widgets:
                self.widgets['rgpd_user_rights'].after_widget_notice = \
                    '<div class="alert-info padding-5">{0}</div>'.format(
                        translate(_("Text samples:<br />"
                                    "- To know and enforce your rights, including the right to "
                                    "withdraw your consent to the use of the data collected by "
                                    "this form, please consult our privacy policy.")))

    def updateGroups(self):
        self.add_group(NamedWidgetsGroup(self, 'head', self.widgets,
                                         ('form_header', 'user_title', 'auth_only',
                                          'submit_label', 'submit_message', 'handler')))
        self.add_group(NamedWidgetsGroup(self, 'captcha', self.widgets,
                                         ('override_captcha', 'client_captcha_key',
                                          'server_captcha_key'),
                                         fieldset_class='inner bordered',
                                         legend=_("Override default captcha settings"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_field=IWfForm['override_captcha']))
        self.add_group(NamedWidgetsGroup(self, 'rgpd', self.widgets,
                                         ('rgpd_consent', 'rgpd_warning', 'rgpd_user_rights'),
                                         fieldset_class='inner bordered',
                                         legend=_("Add RGPD warning"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_field=IWfForm['rgpd_consent']))
        super(FormPropertiesInnerEditForm, self).updateGroups()

    def get_ajax_output(self, changes):
        if 'handler' in changes.get(IWfForm, ()):
            return {
                'status': 'reload',
                'message': self.request.localizer.translate(self.successMessage)
            }
        else:
            return super(FormPropertiesInnerEditForm, self).get_ajax_output(changes)


@subscriber(IDataExtractedEvent, form_selector=FormPropertiesInnerEditForm)
def check_form_properties_data(event):
    """Check form properties input data"""
    data = event.data
    if data.get('rgpd_consent'):
        for attr in ('rgpd_warning', 'rgpd_user_rights'):
            attr_ok = False
            for lang, value in data.get(attr, {}).items():
                if value:
                    attr_ok = True
                    break
            if not attr_ok:
                event.form.widgets.errors += (Invalid(_("You MUST set an RGPD consent text and "
                                                        "RGPD user rights to enable RGPD!")),)
                return


@adapter_config(name='handler-settings',
                context=(IWfForm, IPyAMSLayer, SharedContentPropertiesEditForm),
                provides=IInnerSubForm)
class FormHandlerPropertiesEditForm(InnerAdminEditForm):
    """Form handler properties edit form extension"""

    prefix = 'form_handler.'

    fieldset_class = 'bordered no-x-margin margin-y-10'

    def __new__(cls, context, request, view):
        handler = context.query_handler()
        if handler is None:
            return None
        return InnerAdminEditForm.__new__(cls)

    @property
    def legend(self):
        handler = self.context.query_handler()
        if handler is not None:
            translate = self.request.localizer.translate
            return translate(_("« {handler} » form handler settings")).format(
                handler=translate(handler.label))
        else:
            return _("Form handler settings")

    @property
    def fields(self):
        handler = self.context.query_handler()
        if handler is None:
            interface = Interface
        else:
            interface = handler.handler_info
        return field.Fields(interface)

    weight = 2
